/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.restrictions;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.ui.Callable;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.plugins.odfweb.restrictions.rules.OdfAndRestrictionRule;
import org.ametys.plugins.odfweb.restrictions.rules.OdfMetadataRestrictionRule;
import org.ametys.plugins.odfweb.restrictions.rules.OdfNotRestrictionRule;
import org.ametys.plugins.odfweb.restrictions.rules.OdfOrRestrictionRule;
import org.ametys.plugins.odfweb.restrictions.rules.OdfOrgunitRestrictionRule;
import org.ametys.plugins.odfweb.restrictions.rules.OdfRestrictionRule;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Component able to handle program restrictions related to the "odf-restrictions" site parameter.
 */
public class OdfProgramRestrictionManager extends AbstractLogEnabled implements Component, Serviceable, Contextualizable, Disposable
{
    /** The avalon role. */
    public static final String ROLE = OdfProgramRestrictionManager.class.getName();
    
    /** Site Manager */
    protected SiteManager _siteManager;
    
    /** The content type extension point. */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Root orgunit provider */
    protected RootOrgUnitProvider _rootOrgUnitProvider;
    
    /** Cocoon context */
    protected org.apache.cocoon.environment.Context _cocoonContext;
    
    /** The available odf restrictions */
    protected Map<String, OdfProgramRestriction> _restrictions;
    
    private Context _context;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _rootOrgUnitProvider = (RootOrgUnitProvider) serviceManager.lookup(RootOrgUnitProvider.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void dispose()
    {
        _restrictions = null;
    }
    
    /**
     * Retrieves the available restrictions
     * @return The map of restriction, where keys are restriction ids. 
     */
    public Map<String, OdfProgramRestriction> getRestrictions()
    {
        // Lazy initialize restrictions because orgunit restrictions cannot be
        // resolved during the initialization of the component
        if (_restrictions == null)
        {
            _readRestrictionConfigurationFile();
        }
        
        return _restrictions;
    }
    
    /**
     * Get the ODF restriction for the given ODF root page
     * @param odfRootPage The ODF root page
     * @return The restriction or <code>null</code>
     */
    public OdfProgramRestriction getRestriction(Page odfRootPage)
    {
        Site site = odfRootPage.getSite();
        String restrictionId = site.getValue("odf-restrictions");
        if (StringUtils.isNotEmpty(restrictionId))
        {
            return getRestrictions().get(restrictionId);
        }
        
        return null;
    }
    
    /**
     * Indicate is there is any restriction for this root page
     * @param rootPage odf root page
     * @return true if it is the case
     */
    public boolean hasRestrictions(Page rootPage)
    {
        Site site = rootPage.getSite();
        String restrictionId = site.getValue("odf-restrictions");
        return StringUtils.isNotEmpty(restrictionId) ? getRestrictions().containsKey(restrictionId) : false;
    }
    
    /**
     * Indicate is there is any restriction related to orgunit for this root page
     * @param rootPage odf root page
     * @return true if it is the case
     */
    public boolean hasOrgunitRestrictions(Page rootPage)
    {
        Site site = rootPage.getSite();
        String restrictionId = site.getValue("odf-restrictions");
        if (StringUtils.isNotEmpty(restrictionId))
        {
            OdfProgramRestriction odfProgramRestriction = getRestrictions().get(restrictionId);
            return odfProgramRestriction.hasOrgunitRestrictions();
        }
        
        return false;
    }
    
    /**
     * Get the id of the restriction root orgunit
     * @param siteName the site name
     * @return the id of the restriction root orgunit
     */
    @Callable
    public String getRestrictionRootOrgUnitId(String siteName)
    {
        String odfRootId = null;
        
        Site site = _siteManager.getSite(siteName);
        String restrictionId = site.getValue("odf-restrictions");
        if (StringUtils.isNotEmpty(restrictionId))
        {
            OdfProgramRestriction odfProgramRestriction = getRestrictions().get(restrictionId);
            if (odfProgramRestriction != null && odfProgramRestriction.hasOrgunitRestrictions())
            {
                odfRootId = odfProgramRestriction.getId();
            }
        }
        
        if (StringUtils.isBlank(odfRootId))
        {
            odfRootId = _rootOrgUnitProvider.getRootId();
        }
        
        return odfRootId;
    }
    
    /**
     * Configured the restrictions from the XML configuration file
     */
    protected void _readRestrictionConfigurationFile()
    {
        _restrictions = new HashMap<>();
        
        File restrictionFile = new File (_cocoonContext.getRealPath("/WEB-INF/param/odf-restrictions.xml"));
        
        if (restrictionFile.exists())
        {
            try (InputStream is = new FileInputStream(restrictionFile);)
            {
                Configuration cfg = new DefaultConfigurationBuilder().build(is);
                
                Configuration[] restrictionConfs = cfg.getChildren("restriction");
                for (Configuration restrictionConf : restrictionConfs)
                {
                    String id = restrictionConf.getAttribute("id");
                    I18nizableText label = _configureI18nizableText(restrictionConf, "label", "", "application");
                    List<OdfRestrictionRule> rules = _configureRestrictionRules(restrictionConf.getChild("rules"));
                    
                    _restrictions.put(id, new OdfProgramRestriction(id, label, rules));
                }
                
                Configuration orgunitConf = cfg.getChild("orgunits", false);
                if (orgunitConf != null)
                {
                    _addDefaultOrgunitRestrictions();
                }
            }
            catch (Exception e)
            {
                getLogger().error("Cannot read the configuration file located at /WEB-INF/param/odf-restrictions.xml. Reverting to default.", e);
                _restrictions.clear();
                _addDefaultOrgunitRestrictions();
            }
        }
        else        
        {
            _addDefaultOrgunitRestrictions();
        }
    }
    
    private I18nizableText _configureI18nizableText(Configuration config, String name, String defaultValue, String defaultCatalog)
    {
        Configuration textConfig = config.getChild(name);
        boolean i18nSupported = textConfig.getAttributeAsBoolean("i18n", false);
        String text = textConfig.getValue(defaultValue);
        
        if (i18nSupported)
        {
            String catalogue = textConfig.getAttribute("catalogue", defaultCatalog);
            return new I18nizableText(catalogue, text);
        }
        else
        {
            return new I18nizableText(text);
        }
    }
    
    /**
     * Create a list of rules from configuration nodes
     * @param rulesConf Array of configuration node that represents the set of rules 
     * @return list of rules
     * @throws ConfigurationException if a configuration error is encountered
     */
    protected List<OdfRestrictionRule> _configureRestrictionRules(Configuration rulesConf) throws ConfigurationException
    {
        List<OdfRestrictionRule> rules = new ArrayList<>();
        
        for (Configuration ruleConf : rulesConf.getChildren())
        {
            rules.add(_configureRestrictionRule(ruleConf));
        }
        
        return rules;
    }
    
    /**
     * Configure a restriction rule from a configuration node
     * @param ruleConf The configuration node representing the rule
     * @return The odf restriction rule
     * @throws ConfigurationException if a configuration error is encountered
     */
    protected OdfRestrictionRule _configureRestrictionRule(Configuration ruleConf) throws ConfigurationException
    {
        String name = ruleConf.getName();
        
        if ("metadata".equals(name))
        {
            String metadataPath = ruleConf.getAttribute("name");
            String metadataValue = ruleConf.getAttribute("value");
            return new OdfMetadataRestrictionRule(metadataPath, metadataValue);
        }
        else if ("orgunit".equals(name))
        {
            String orgunitId = ruleConf.getAttribute("id", null);
            
            if (StringUtils.isEmpty(orgunitId))
            {
                throw new ConfigurationException("Expecting 'id' attribute for orgunit restriction rule.");
            }
            
            return new OdfOrgunitRestrictionRule(_rootOrgUnitProvider, orgunitId);
        }
        else if ("and".equals(name))
        {
            List<OdfRestrictionRule> childRules = _configureRestrictionRules(ruleConf);
            return new OdfAndRestrictionRule(childRules);
        }
        else if ("or".equals(name))
        {
            List<OdfRestrictionRule> childRules = _configureRestrictionRules(ruleConf);
            return new OdfOrRestrictionRule(childRules);
        }
        else if ("not".equals(name))
        {
            List<OdfRestrictionRule> childRules = _configureRestrictionRules(ruleConf);
            return new OdfNotRestrictionRule(childRules);
        }
        
        throw new ConfigurationException("Unknow node name in restriction configuration : " + name);
    }
    
    private void _addDefaultOrgunitRestrictions()
    {
        Request request = ContextHelper.getRequest(_context);
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        try
        {
            // Force default workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, null);
            
            String rootOrgunitId = _rootOrgUnitProvider.getRootId();
            Set<String> orgunitIds = _rootOrgUnitProvider.getChildOrgUnitIds(rootOrgunitId, true);
            orgunitIds.add(rootOrgunitId);
            
            for (String id : orgunitIds)
            {
                if (!StringUtils.equals(id, rootOrgunitId))
                {
                    OrgUnit orgunit = _resolver.resolveById(id);
                    
                    OdfRestrictionRule rule = new OdfOrgunitRestrictionRule(_rootOrgUnitProvider, id);
                    OdfProgramRestriction restriction = new OdfProgramRestriction(id, new I18nizableText(orgunit.getTitle()), Collections.singletonList(rule));
                    
                    _restrictions.put(id, restriction);
                }
            }
        }
        finally
        {
            // Restore workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
}
