/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.clientside;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.value.StringValue;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.util.LambdaUtils;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.plugins.odfweb.OdfWebObservationConstants;
import org.ametys.plugins.odfweb.repository.FirstLevelPageFactory;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;

import com.google.common.collect.ObjectArrays;

/**
 * This element creates an action button to set the ODF root page
 */
public class ODFRootClientSideElement extends StaticClientSideElement
{
    /** Ametys resolver */
    protected AmetysObjectResolver _resolver;
    /** Catalog manager */
    protected CatalogsManager _catalogsManager;
    /** Odf page handler */
    protected OdfPageHandler _odfPageHandler;
    /** Observation manager */
    protected ObservationManager _observationManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _catalogsManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
        _odfPageHandler = (OdfPageHandler) smanager.lookup(OdfPageHandler.ROLE);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
    }
    
    /**
     * Get odf root page status for the current page
     * @param pageId the page identifier
     * @return A map representing its status
     */
    @Callable
    public Map<String, Object> getStatus(String pageId)
    {
        Map<String, Object> results = new HashMap<>();
        Page page = _resolver.resolveById(pageId);
        
        // page id
        results.put("page-id", page.getId());
        
        if (page instanceof JCRAmetysObject)
        {
            if (_odfPageHandler.isODFRootPage(page))
            {
                results.put("is-odf-root", true);
                
                // Odf root page description
                I18nizableText rawDescription = (I18nizableText) this._script.getParameters().get("odf-root-page-description");
                Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
                i18nParams.put("title", new I18nizableText(page.getTitle()));
                
                I18nizableText description = new I18nizableText(rawDescription.getCatalogue(), rawDescription.getKey(), i18nParams);
                results.put("odf-root-page-description", description);
                
                // remove odf root
                rawDescription = (I18nizableText) this._script.getParameters().get("remove-odf-page-description");
                description = new I18nizableText(rawDescription.getCatalogue(), rawDescription.getKey(), i18nParams);
                results.put("remove-odf-page-description", description);
                
                // Catalog Odf root page descrption
                String catalogCode = page.getValue(OdfPageHandler.CATALOG_DATA_NAME);
                if (StringUtils.isNotEmpty(catalogCode))
                {
                    Catalog catalog = _catalogsManager.getCatalog(catalogCode);
                    I18nizableText catalogText = new I18nizableText(catalog != null ? catalog.getTitle() : catalogCode);
                    
                    i18nParams = new HashMap<>();
                    i18nParams.put("catalog", catalogText);
                    
                    rawDescription = (I18nizableText) this._script.getParameters().get("catalog-description");
                    description = new I18nizableText(rawDescription.getCatalogue(), rawDescription.getKey(), i18nParams);
                    results.put("catalog-description", description);
                }
            }
            else
            {
                // Add odf root page
                results.put("add-odf-page", true);
                
                I18nizableText rawDescription = (I18nizableText) this._script.getParameters().get("add-odf-page-description");
                Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
                i18nParams.put("title", new I18nizableText(page.getTitle()));
                
                I18nizableText description = new I18nizableText(rawDescription.getCatalogue(), rawDescription.getKey(), i18nParams);
                results.put("add-odf-page-description", description);
            }
        }
        else
        {
            // Invalid page
            results.put("invalid-page", true);
            
            I18nizableText rawDescription = (I18nizableText) this._script.getParameters().get("no-jcr-page-description");
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
            i18nParams.put("title", new I18nizableText(page.getTitle()));
            
            I18nizableText description = new I18nizableText(rawDescription.getCatalogue(), rawDescription.getKey(), i18nParams);
            results.put("no-jcr-page-description", description);
        }
        
        return results;
    }
    
    /**
     * Retrieves odf root page properties 
     * @param pageId page identifier
     * @return the map of properties
     */
    @Callable
    public Map<String, Object> getOdfRootPageInfo(String pageId)
    {
        Map<String, Object> properties = new HashMap<>();
        
        Page page = _resolver.resolveById(pageId);
        Set<Page> currentODFPages = _odfPageHandler.getOdfRootPages(page.getSiteName(), page.getSitemapName());
                
        if (_catalogsManager.getCatalogs().isEmpty())
        {
            properties.put("noCatalog", true);
        }
        
        if (!currentODFPages.contains(page))
        {
            properties.put("isOdfRootPage", false);
        }
        else
        {
            properties.put("isOdfRootPage", true);
            
            properties.put("catalog", page.getValue(OdfPageHandler.CATALOG_DATA_NAME, ""));
            properties.put("firstLevel", page.getValue(OdfPageHandler.LEVEL1_ATTRIBUTE_NAME, ""));
            properties.put("secondLevel", page.getValue(OdfPageHandler.LEVEL2_ATTRIBUTE_NAME, ""));
        }
        
        return properties;
    }
    
    /**
     * Add or update the odf root page property on a page
     * @param pageId the page identifier
     * @param catalog the catalog to set
     * @param firstLevel the first level to set
     * @param secondLevel the second level to set
     * @throws RepositoryException if a repository error occurs
     * @return true if the page has actually been modified
     */
    @Callable
    public boolean addOrUpdateOdfRootProperty(String pageId, String catalog, String firstLevel, String secondLevel) throws RepositoryException
    {
        Page page = _resolver.resolveById(pageId);
        Set<Page> currentODFPages = _odfPageHandler.getOdfRootPages(page.getSiteName(), page.getSitemapName());
        
        if (!currentODFPages.contains(page))
        {
            return _addOdfRootProperty(page, catalog, firstLevel, secondLevel);
        }
        else
        {
            return _updateOdfRootProperty(page, catalog, firstLevel, secondLevel);
        }
    }
    
    private boolean _addOdfRootProperty(Page page, String catalog, String firstLevel, String secondLevel) throws RepositoryException
    {
        boolean hasChanges = false;
        
        if (page instanceof JCRAmetysObject)
        {
            JCRAmetysObject jcrPage = (JCRAmetysObject) page;
            
            // Add odf root property
            Property virtualProperty = _getVirtualProperty(jcrPage);
            Value[] oldValues = virtualProperty != null ? virtualProperty.getValues() : new Value[]{};
            
            boolean hasVirtualProperty = Arrays.stream(oldValues)
                    .anyMatch(LambdaUtils.wrapPredicate(ODFRootClientSideElement::isVirtualPageFactoryValue));
            
            if (!hasVirtualProperty)
            {
                Value[] newValues = ObjectArrays.concat(oldValues, new StringValue(FirstLevelPageFactory.class.getName()));
                _setVirtualProperty(jcrPage, newValues);
            }
            
            ModifiablePage modifiablePage = (ModifiablePage) page;
            
            // Set the ODF catalog property
            if (StringUtils.isNotEmpty(catalog))
            {
                modifiablePage.setValue(OdfPageHandler.CATALOG_DATA_NAME, catalog);
            }
            
            // Set the level properties.
            if (StringUtils.isNotEmpty(firstLevel))
            {
                modifiablePage.setValue(OdfPageHandler.LEVEL1_ATTRIBUTE_NAME, firstLevel);
            }
            else
            {
                modifiablePage.removeValue(OdfPageHandler.LEVEL1_ATTRIBUTE_NAME);
            }
            
            if (StringUtils.isNotEmpty(secondLevel))
            {
                modifiablePage.setValue(OdfPageHandler.LEVEL2_ATTRIBUTE_NAME, secondLevel);
            }
            else
            {
                modifiablePage.removeValue(OdfPageHandler.LEVEL2_ATTRIBUTE_NAME);
            }
            
            if (jcrPage.needsSave())
            {
                jcrPage.saveChanges();
                hasChanges = true;
            }
        }
        
        if (hasChanges)
        {
            _notifyOdfRootPageChange(page);
        }
        
        return hasChanges;
    }
    
    private boolean _updateOdfRootProperty(Page page, String catalog, String firstLevel, String secondLevel)
    {
        boolean hasChanges = false;
        
        if (page instanceof ModifiablePage)
        {
            ModifiablePage modifiablePage = (ModifiablePage) page;
            
            // Set the ODF catalog property
            if (StringUtils.isNotEmpty(catalog))
            {
                modifiablePage.setValue(OdfPageHandler.CATALOG_DATA_NAME, catalog);
            }
            
            // Set the level properties.
            if (StringUtils.isNotEmpty(firstLevel))
            {
                modifiablePage.setValue(OdfPageHandler.LEVEL1_ATTRIBUTE_NAME, firstLevel);
            }
            else
            {
                modifiablePage.removeValue(OdfPageHandler.LEVEL1_ATTRIBUTE_NAME); //TODO level 1 mandatory ??
            }
            
            if (StringUtils.isNotEmpty(secondLevel))
            {
                modifiablePage.setValue(OdfPageHandler.LEVEL2_ATTRIBUTE_NAME, secondLevel);
            }
            else
            {
                modifiablePage.removeValue(OdfPageHandler.LEVEL2_ATTRIBUTE_NAME);
            }
            
            if (modifiablePage.needsSave())
            {
                modifiablePage.saveChanges();
                hasChanges = true;
            }
        }
        
        if (hasChanges)
        {
            // Odf root updated observer
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_PAGE, page);
            _observationManager.notify(new Event(OdfWebObservationConstants.ODF_ROOT_UPDATED, _currentUserProvider.getUser(), eventParams));
            
            _notifyOdfRootPageChange(page);
        }
        
        return hasChanges;
    }
    
    /**
     * Remove the odf root page property on a page
     * @param pageId the page identifier
     * @return true if the property has actually been removed
     * @throws RepositoryException if a repository error occurs
     */
    @Callable
    public boolean removeOdfRootPage(String pageId) throws RepositoryException
    {
        boolean hasChanges = false;
        Page page = _resolver.resolveById(pageId);
        
        if (page instanceof JCRAmetysObject)
        {
            JCRAmetysObject jcrPage = (JCRAmetysObject) page;
            Property virtualProperty = _getVirtualProperty(jcrPage);
            
            if (virtualProperty != null)
            {
                Value[] oldValues = virtualProperty.getValues();
                
                // remove the virtual page property
                Value[] newValues = Arrays.stream(oldValues)
                    .filter(LambdaUtils.wrapPredicate(ODFRootClientSideElement::isVirtualPageFactoryValue).negate())
                    .toArray(Value[]::new);
                
                _setVirtualProperty(jcrPage, newValues);
                
                // Remove the level properties.
                ModifiablePage modifiablePage = (ModifiablePage) page;
                modifiablePage.removeValue(OdfPageHandler.LEVEL1_ATTRIBUTE_NAME);
                modifiablePage.removeValue(OdfPageHandler.LEVEL2_ATTRIBUTE_NAME);
                modifiablePage.removeValue(OdfPageHandler.CATALOG_DATA_NAME);
                
                if (jcrPage.needsSave())
                {
                    jcrPage.saveChanges();
                    hasChanges = true;
                }
            }
        }
        
        if (hasChanges)
        {
            // Odf root deleted observer
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_PAGE, page);
            _observationManager.notify(new Event(OdfWebObservationConstants.ODF_ROOT_DELETED, _currentUserProvider.getUser(), eventParams));
            
            _notifyOdfRootPageChange(page);
        }
        
        return hasChanges;
    }
    
    private void _notifyOdfRootPageChange(Page page)
    {
        // clear root cache
        _odfPageHandler.clearRootCache();
        
        // Page changed observer
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_PAGE, page);
        _observationManager.notify(new Event(ObservationConstants.EVENT_PAGE_CHANGED, _currentUserProvider.getUser(), eventParams));
    }
    
    private Property _getVirtualProperty(JCRAmetysObject jcrPage) throws RepositoryException
    {
        Node node = jcrPage.getNode();
        
        if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
        {
            return node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY);
        }
        
        return null;
    }
    
    private void _setVirtualProperty(JCRAmetysObject jcrPage, Value[] values) throws RepositoryException
    {
        jcrPage.getNode().setProperty(AmetysObjectResolver.VIRTUAL_PROPERTY, values);
    }
    
    /**
     * Tests if a JCR Value represents the virtual page property
     * @param value the value to test
     * @return true if it is the case
     * @throws RepositoryException if a repository error occurs
     */
    private static boolean isVirtualPageFactoryValue(Value value) throws RepositoryException 
    {
        return StringUtils.equals(value.getString(), FirstLevelPageFactory.class.getName());
    }
}
