/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.odfweb.service.search.criterion.DegreeUniversityAttributeContentSearchCriterionDefinition;
import org.ametys.plugins.odfweb.service.search.helper.DegreeUniversityHelper;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.web.frontoffice.search.requesttime.SearchComponent;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.frontoffice.search.requesttime.input.SearchUserInputs;

/**
 * {@link SearchComponent} for handle degree university inputs
 */
public class DegreeUniversitySearchComponent implements SearchComponent, Serviceable
{
    /** The degree university helper */
    protected DegreeUniversityHelper _degreeUniversityHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _degreeUniversityHelper = (DegreeUniversityHelper) manager.lookup(DegreeUniversityHelper.ROLE);
    }
    
    @Override
    public int priority()
    {
        return MAX_PRIORITY;
    }

    @Override
    public boolean supports(SearchComponentArguments args)
    {
        return args.launchSearch();
    }

    @Override
    public void execute(SearchComponentArguments args) throws Exception
    {
        SearchUserInputs userInputs = args.userInputs();
        
        Map<String, List<String>> facets = new HashMap<>();
        for (Entry<String, List<String>> entry : userInputs.facets().entrySet())
        {
            if (entry.getKey().equals("ContentReturnable$" + DegreeUniversityAttributeContentSearchCriterionDefinition.DEGREE_UNIVERSITY_SEARCH_CRITERION_ID))
            {
                facets.put(_getCriterionKey(args), _updatedFacetsValues(entry.getValue()));
            }
            else
            {
                facets.put(entry.getKey(), entry.getValue());
            }
        }
        userInputs.setFacets(facets);
        
        args.updateUserInputs(userInputs);
    }
    
    private String _getCriterionKey(SearchComponentArguments args)
    {
        String key = "ContentReturnable$" + DegreeUniversityAttributeContentSearchCriterionDefinition.DEGREE_SEARCH_CRITERION_ID;
        boolean present = args.serviceInstance()
            .getFacets()
            .stream()
            .anyMatch(f -> f.getId().equals(key));
        
        return present ? key : "ContentReturnable$" + DegreeUniversityAttributeContentSearchCriterionDefinition.DEGREE_UNIVERSITY_SEARCH_CRITERION_ID;
    }
    
    private List<String> _updatedFacetsValues(List<String> facets)
    {
        return facets.stream()
            .map(this::_getDegreeValues)
            .flatMap(List::stream)
            .collect(Collectors.toList());
    }
    
    private List<String> _getDegreeValues(String value)
    {
        if (DegreeUniversityAttributeContentSearchCriterionDefinition.SEARCH_CRITERION_ALL_DU_VALUE.equals(value))
        {
            return _degreeUniversityHelper.getDegrees(true)
                .stream()
                .map(AmetysObject::getId)
                .collect(Collectors.toList());
        }
        
        return List.of(value);
    }
}
