/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.cms.search.SearchResults;
import org.ametys.plugins.odfweb.service.search.criterion.DegreeUniversityAttributeContentSearchCriterionDefinition;
import org.ametys.plugins.odfweb.service.search.helper.DegreeUniversityHelper;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstance;
import org.ametys.web.frontoffice.search.instance.model.FOSearchCriterion;
import org.ametys.web.frontoffice.search.metamodel.FacetDefinition;
import org.ametys.web.frontoffice.search.requesttime.SearchComponent;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.frontoffice.search.requesttime.impl.SaxEnumeratedCriteriaComponent;
import org.ametys.web.frontoffice.search.requesttime.impl.SearchComponentHelper;

/**
 * {@link SearchComponent} for saxing number of results for degree university criteria
 */
public class SaxDegreeUniversityEnumeratedCriteriaComponent extends SaxEnumeratedCriteriaComponent
{
    /** The degree university helper */
    protected DegreeUniversityHelper _degreeUniversityHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _degreeUniversityHelper = (DegreeUniversityHelper) manager.lookup(DegreeUniversityHelper.ROLE);
    }
    
    @Override
    public int priority()
    {
        return super.priority() + 1;
    }
    
    @Override
    public void execute(SearchComponentArguments args) throws Exception
    {
        Optional<SearchResults<AmetysObject>> enumeratedResults = args.enumeratedResults();
        // Search for enumerated criteria is alreay done, just sax the search results for degree university criterion
        if (enumeratedResults.isPresent())
        {
            ContentHandler contentHandler = args.contentHandler();
            XMLUtils.startElement(contentHandler, "enumerated-criteria");
            
            Map<String, Object> contextualParameters = SearchComponentHelper.getSearchComponentContextualParameters(args);
            Map<FacetDefinition, FOSearchCriterion> serviceFacets = _getFacetDefinitions(args.serviceInstance());
            _saxCountEnumeratedCriteria(contentHandler, args.generatorParameters(), serviceFacets, enumeratedResults.get(), contextualParameters);
            
            XMLUtils.endElement(contentHandler, "enumerated-criteria");
        }
        else
        {
            super.execute(args);
        }
    }
    
    @Override
    protected Map<FacetDefinition, FOSearchCriterion> _getFacetDefinitions(SearchServiceInstance serviceInstance)
    {
        // Get only facet for DU
        return super._getFacetDefinitions(serviceInstance)
                    .entrySet()
                    .stream()
                    .filter(entry -> entry.getKey()
                                          .getId()
                                          .equals("ContentReturnable$" + DegreeUniversityAttributeContentSearchCriterionDefinition.DEGREE_UNIVERSITY_SEARCH_CRITERION_ID))
                    .collect(Collectors.toMap(
                            Map.Entry::getKey,
                            Map.Entry::getValue));
    }
    
    @Override
    protected void _saxFacetItemsWithCount(ContentHandler contentHandler, FacetDefinition facet, Map<String, Integer> valuesForCurrentFacetDef, Map<String, Object> contextualParameters) throws SAXException
    {
        // Get DU ids
        List<String> dUIds = _degreeUniversityHelper.getDegrees(true)
                .stream()
                .map(Content::getId)
                .collect(Collectors.toList());
        
        // Sum all facet count for DU values
        int allCount = 0;
        for (String value : valuesForCurrentFacetDef.keySet())
        {
            if (dUIds.contains(value))
            {
                allCount += valuesForCurrentFacetDef.get(value);
            }
        }
        
        // Sax facet for "ALL DU" value
        Map<String, Integer> valuesForAllDU = Map.of(DegreeUniversityAttributeContentSearchCriterionDefinition.SEARCH_CRITERION_ALL_DU_VALUE, allCount);
        super._saxFacetItemsWithCount(contentHandler, facet, valuesForAllDU, contextualParameters);
    }
}
