/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.xslt;

import java.util.List;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.repository.Content;
import org.ametys.odf.EducationalPathHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.skill.ODFSkillsHelper;
import org.ametys.plugins.odfweb.repository.CoursePage;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.odfweb.repository.ProgramPage;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.transformation.xslt.AmetysXSLTHelper;

/**
 * Helper component to be used from XSL stylesheets.
 */
public class OdfXSLTHelper extends org.ametys.odf.OdfXSLTHelper implements Contextualizable
{
    /** The ODF page handler */
    protected static OdfPageHandler _odfPageHandler;
    
    /** The ODF skills helper */
    protected static ODFSkillsHelper _odfSkillsHelper;
    
    /** The avalon context */
    protected static Context _context;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfPageHandler = (OdfPageHandler) smanager.lookup(OdfPageHandler.ROLE);
        _odfSkillsHelper = (ODFSkillsHelper) smanager.lookup(ODFSkillsHelper.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the ODF root page, for a specific site, language.
     * If there is many ODF root pages, the first page of the list is returned.
     * @param siteName the desired site name.
     * @param language the sitemap language to search in.
     * @return the first ODF root page, or null if not found
     */
    public static String odfRootPage(String siteName, String language)
    {
        Page odfRootPage = _odfPageHandler.getOdfRootPage(siteName, language);
        if (odfRootPage != null)
        {
            return odfRootPage.getId();
        }
        return null;
    }
    
    /**
     * Get the ODF root page, for a specific site, language and catalog.
     * @param siteName the desired site name.
     * @param language the sitemap language to search in.
     * @param catalog The ODF catalog
     * @return the ODF root page, or null if not found
     */
    public static String odfRootPage(String siteName, String language, String catalog)
    {
        Page odfRootPage = _odfPageHandler.getOdfRootPage(siteName, language, catalog);
        if (odfRootPage != null)
        {
            return odfRootPage.getId();
        }
        return null;
    }
    
    /**
     * Get the PDF url of a program or a subprogram
     * @param contentId The content id
     * @param siteName The site name
     * @return the PDF url or empty string if the content is not a {@link Program} or {@link SubProgram}
     */
    public static String odfPDFUrl (String contentId, String siteName)
    {
        StringBuilder sb = new StringBuilder();
        
        Content content = _ametysObjectResolver.resolveById(contentId);
        if (content instanceof AbstractProgram)
        {
            sb.append(AmetysXSLTHelper.uriPrefix())
                .append("/plugins/odf-web/")
                .append(siteName)
                .append("/_content/")
                .append(content.getName())
                .append(".pdf");
        }
        
        return sb.toString();
    }

    /**
     * Get the id of parent program from the current page
     * @return the id of parent program or null if not found
     */
    public static String parentProgramId()
    {
        String pageId = AmetysXSLTHelper.pageId();
        
        if (StringUtils.isNotEmpty(pageId))
        {
            Page page = _ametysObjectResolver.resolveById(pageId);
            
            AmetysObject parent = page.getParent();
            while (!(parent instanceof Sitemap))
            {
                if (parent instanceof ProgramPage)
                {
                    return ((ProgramPage) parent).getProgram().getId();
                }
                
                parent = parent.getParent();
            }
        }
        
        return null;
    }
    
    /**
     * Get the ECTS of the current course for the current context if present
     * @return the ECTS or 0 if not found
     */
    public static double getCurrentEcts()
    {
        Pair<ProgramItem, List<EducationalPath>> currentEducationalPaths = _getCurrentEducationalPaths();
        if (currentEducationalPaths != null)
        {
            ProgramItem programItem = currentEducationalPaths.getLeft();
            if (programItem instanceof Course course)
            {
                return course.getEcts(currentEducationalPaths.getRight());
            }
        }
        
        return 0;
    }
    
    private static Pair<ProgramItem, List<EducationalPath>> _getCurrentEducationalPaths()
    {
        Request request = ContextHelper.getRequest(_context);
        Page page = (Page) request.getAttribute(WebConstants.REQUEST_ATTR_PAGE);
        
        // First try to get current educational paths from course page if present
        if (page != null)
        {
            if (page instanceof CoursePage coursePage)
            {
                Course course = coursePage.getContent();
                return Pair.of(course, course.getCurrentEducationalPaths());
            }
            else if (page instanceof ProgramPage programPage)
            {
                AbstractProgram abstractProgram = programPage.getContent();
                return Pair.of(abstractProgram, abstractProgram.getCurrentEducationalPaths());
            }
        }
        
        // Then try to get current educational paths from course content if present
        Content content = (Content) request.getAttribute(Content.class.getName());
        return _getCurrentEducationalPaths(content);
    }
    
    private static Pair<ProgramItem, List<EducationalPath>> _getCurrentEducationalPaths(Content content)
    {
        Request request = ContextHelper.getRequest(_context);
        if (content != null && (content instanceof Course || content instanceof AbstractProgram))
        {
            // First try to get educational paths from content
            List<EducationalPath> currentEducationalPaths = content instanceof Course course ? course.getCurrentEducationalPaths() : ((AbstractProgram) content).getCurrentEducationalPaths();
            if (currentEducationalPaths == null)
            {
                // If null try to get educational paths from request attributes
                @SuppressWarnings("unchecked")
                List<ProgramItem> pathFromRequest = (List<ProgramItem>) request.getAttribute(EducationalPathHelper.PROGRAM_ITEM_ANCESTOR_PATH_REQUEST_ATTR);
                if (pathFromRequest != null)
                {
                    // In request the path may be a partial path
                    currentEducationalPaths = _odfHelper.getEducationPathFromPath(pathFromRequest);
                    
                    // If ancestor is present in request attribute, filter paths that contains this ancestor
                    ProgramItem ancestor = (ProgramItem) request.getAttribute(EducationalPathHelper.ROOT_PROGRAM_ITEM_REQUEST_ATTR);
                    if (ancestor != null)
                    {
                        currentEducationalPaths = currentEducationalPaths.stream()
                                .filter(p -> p.getProgramItemIds().contains(ancestor.getId()))
                                .toList();
                    }
                }
                else
                {
                    // Cannot determine current educational paths from context, returns all available education paths
                    currentEducationalPaths = _odfHelper.getEducationalPaths((ProgramItem) content, true, true);
                }
            }
            
            return Pair.of((ProgramItem) content, currentEducationalPaths);
        }
        
        return null;
    }
    
    /**
     * Get the ECTS of the current course for the current context if present
     * @param defaultValue The default value
     * @return the ECTS or 0 if not found
     */
    public static double getCurrentEcts(String defaultValue)
    {
        double currentEcts = getCurrentEcts();
        return currentEcts != 0 ? currentEcts : (StringUtils.isNotEmpty(defaultValue) ? Double.valueOf(defaultValue) : 0);
    }
    
    /**
     * Determines if the values of ECTS is equals for the course's educational paths in the current context
     * @param courseId The course id
     * @return true if the values of ECTS is equals in the current context
     */
    public static boolean areECTSEqual(String courseId)
    {
        Course course = _ametysObjectResolver.resolveById(courseId);
        return _areECTSEqual(course);
    }
    
    /**
     * Determines if the values of ECTS is equals for the current course's educational paths in the current context
     * @return true if the values of ECTS is equals in the current context
     */
    public static boolean areECTSEqual()
    {
        Request request = ContextHelper.getRequest(_context);
        Content content = (Content) request.getAttribute(Content.class.getName());
        if (content != null && content instanceof Course course)
        {
            return _areECTSEqual(course);
        }
        return false;
    }
    
    private static boolean _areECTSEqual(Course course)
    {
        Pair<ProgramItem, List<EducationalPath>> currentEducationalPaths = _getCurrentEducationalPaths(course);
        if (currentEducationalPaths != null)
        {
            ProgramItem programItem = currentEducationalPaths.getLeft();
            List<EducationalPath> paths = currentEducationalPaths.getRight();
            return paths != null ? _odfHelper.isSameValueForPaths(programItem, Course.ECTS_BY_PATH, paths) : _odfHelper.isSameValueForAllPaths(programItem, Course.ECTS_BY_PATH);
        }
        else
        {
            return _odfHelper.isSameValueForAllPaths(course, Course.ECTS_BY_PATH);
        }
    }
    
    /**
     * <code>true</code> if the program item is part of an program item (program, subprogram or container) that is excluded from skills
     * @param programItemId the program item id
     * @param programPageItemId the program item page id. If null or empty, program item is display with no context, consider that skills are available
     * @return <code>true</code> if the program item has an excluded parent in it path from the page context
     */
    public static boolean areSkillsUnavailable(String programItemId, String programPageItemId)
    {
        if (StringUtils.isBlank(programItemId) || StringUtils.isBlank(programPageItemId))
        {
            // program part is displayed outside a page context, assuming that skills should be displayed
            return false;
        }
        
        ProgramItem programItem = _ametysObjectResolver.resolveById(programItemId);
        if (programItem instanceof Program)
        {
            return _odfSkillsHelper.isExcluded(programItem);
        }
        
        Page programItemPage = _ametysObjectResolver.resolveById(programPageItemId);
        
        ProgramPage closestProgramPage = _getClosestProgramPage(programItemPage);
        AbstractProgram closestProgramOrSubprogram = closestProgramPage.getProgram();
        
        ProgramItem parent = _odfHelper.getParentProgramItem(programItem, closestProgramOrSubprogram);
        while (parent != null && !(parent instanceof Program))
        {
            if (_odfSkillsHelper.isExcluded(parent))
            {
                // If the parent is excluded, the skills are unavailable
                return true;
            }
            
            // If the closest program parent is a subprogram, continue to its program parent
            if (closestProgramOrSubprogram instanceof SubProgram && closestProgramOrSubprogram.equals(parent))
            {
                closestProgramOrSubprogram = ((ProgramPage) closestProgramPage.getParent()).getProgram();
            }
            parent = _odfHelper.getParentProgramItem(parent, closestProgramOrSubprogram);
        }
        
        return parent != null ? _odfSkillsHelper.isExcluded(parent) : false;
    }
    
    private static ProgramPage _getClosestProgramPage(Page page)
    {
        Page parentPage = page.getParent();
        while (!(parentPage instanceof ProgramPage))
        {
            parentPage = parentPage.getParent();
        }
        
        return (ProgramPage) parentPage;
    }
}
