/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.actions;

import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ResourceNotFoundException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.plugins.odfweb.repository.CoursePage;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.config.Config;
import org.ametys.web.WebHelper;

/**
 * Get a course page from its code. Needed for IPWeb.
 */
public class GetCourseForIPWebAction extends ServiceableAction
{
    private CatalogsManager _catalogsManager;
    private OdfPageResolver _odfPageResolver;
    private ContentTypeExtensionPoint _contentTypeEP;
    private AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _catalogsManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
        _odfPageResolver = (OdfPageResolver) smanager.lookup(OdfPageResolver.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        // Check if IPWeb is activated in the configuration
        if (!Config.getInstance().getValue("odf.ipweb.activate", true, false))
        {
            throw new UnsupportedOperationException("The IPWeb URL resolution is not configured.");
        }
        
        String field = getField();
        String lang = getLanguage();
        String catalog = getCatalog();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        // Get course from code
        String courseCode = request.getParameter("CodElp");
        Course course = searchCourse(catalog, lang, courseCode, field);

        // Get the page
        String url = resolvePage(request, course, field);
        redirector.redirect(false, url);
        
        return EMPTY_MAP;
    }
    
    private String getField() throws ResourceNotFoundException
    {
        String field = Config.getInstance().getValue("odf.ipweb.field");
        
        // Check if the configured field is valid
        if (!_contentTypeEP.getExtension(CourseFactory.COURSE_CONTENT_TYPE).hasModelItem(field))
        {
            throw new ResourceNotFoundException(String.format("The field '%s' is not defined on course type.", field));
        }
        
        return field;
    }
    
    private String getLanguage()
    {
        return Optional.of("odf.ipweb.lang")
                .map(Config.getInstance()::<String>getValue)
                .filter(StringUtils::isNotEmpty)
                .orElseGet(() -> Config.getInstance().getValue("odf.programs.lang"));
    }
    
    private String getCatalog() throws ResourceNotFoundException
    {
        String catalog = Optional.of("odf.ipweb.catalog")
                .map(Config.getInstance()::<String>getValue)
                .filter(StringUtils::isNotEmpty)
                .orElseGet(_catalogsManager::getDefaultCatalogName);
        
        // Check if the catalog is valid
        if (_catalogsManager.getCatalog(catalog) == null)
        {
            throw new ResourceNotFoundException(String.format("The catalog '%s' does not exist", catalog));
        }
        
        return catalog;
    }
    
    private Course searchCourse(String catalog, String lang, String code, String fieldCode) throws ResourceNotFoundException
    {
        AndExpression expression = new AndExpression();
        expression.add(new ContentTypeExpression(Operator.EQ, CourseFactory.COURSE_CONTENT_TYPE));
        expression.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        expression.add(new LanguageExpression(Operator.EQ, lang));
        expression.add(new StringExpression(fieldCode, Operator.EQ, code));
        
        String xpathQuery = ContentQueryHelper.getContentXPathQuery(expression);
        AmetysObjectIterable<Course> courses = _resolver.query(xpathQuery);
        
        if (courses.getSize() > 1)
        {
            throw new ResourceNotFoundException(String.format("There are several courses with %s '%s' in catalog '%s' with language '%s'", fieldCode, code, catalog, lang));
        }
        
        Course course = courses.stream().findFirst().orElse(null);
        if (course == null)
        {
            throw new ResourceNotFoundException(String.format("There is no course with %s '%s' in catalog '%s' with language '%s'", fieldCode, code, catalog, lang));
        }
        
        return course;
    }
    
    private String resolvePage(Request request, Course course, String field) throws ResourceNotFoundException
    {
        String siteName = WebHelper.getSiteName(request);
        
        // Get the page
        CoursePage page = _odfPageResolver.getCoursePage(course, (AbstractProgram) null, siteName);
        if (page == null)
        {
            throw new ResourceNotFoundException(String.format("There is no page for course with code '%s' in catalog '%s' with language '%s' on site '%s'", course.<String>getValue(field), course.getCatalog(), course.getLanguage(), siteName));
        }
        
        return ResolveURIComponent.resolve("page", page.getId(), false, true, false);
    }
}
