/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.tree.OdfClassificationHandler.LevelValue;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectFactoryExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.provider.WorkspaceSelector;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;

import com.google.common.collect.Iterables;

/**
 * Maintains a per-request cache, dispatching ODF virtual pages accross degrees and domains.
 */
public class ODFPageCache extends AbstractLogEnabled implements Serviceable, Initializable, Component
{
    /** Avalon role. */
    public static final String ROLE = ODFPageCache.class.getName();
    
    // Tree key when no first level metadata has been selected
    private static final String __NO_FIRST_DATA_KEY = "_no-first-data";
    
    // Tree key when no second level metadata has been selected
    private static final String __NO_SECOND_DATA_KEY = "_no-second-data";
    
    private static final String __TREE_CACHE = ODFPageCache.class.getName() + "$tree";
    private static final String __PROGRAM_CACHE = ODFPageCache.class.getName() + "$program";
    
    private OdfPageHandler _odfPageHandler;
    private WorkspaceSelector _workspaceSelector;
    private AmetysObjectResolver _resolver;

    private AmetysObjectFactoryExtensionPoint _ametysObjectFactoryEP;

    private AbstractCacheManager _cacheManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfPageHandler = (OdfPageHandler) manager.lookup(OdfPageHandler.ROLE);
        _workspaceSelector = (WorkspaceSelector) manager.lookup(WorkspaceSelector.ROLE);
        _ametysObjectFactoryEP = (AmetysObjectFactoryExtensionPoint) manager.lookup(AmetysObjectFactoryExtensionPoint.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createRequestCache(__TREE_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ODF_PAGE_TREE_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ODF_PAGE_TREE_DESCRIPTION"),
                false);
        _cacheManager.createRequestCache(__PROGRAM_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ODF_PAGE_PROGRAM_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ODF_PAGE_PROGRAM_DESCRIPTION"),
                false);
    }
    
    Map<String, Map<String, Collection<Program>>> getProgramCache(Page rootPage, boolean computeIfNotPresent)
    {
        String workspace = _workspaceSelector.getWorkspace();
        String pageId = rootPage.getId();
        
        Cache<ODFPageCacheKey, Map<String, Map<String, Collection<Program>>>> odfPageCache = _getODFPageCache();
        Map<String, Map<String, Collection<Program>>> level1Tree = odfPageCache.get(ODFPageCacheKey.of(workspace, pageId));
        
        if (level1Tree == null)
        {
            if (!computeIfNotPresent)
            {
                return null;
            }
            
            Map<String, LevelValue> level1Values = _odfPageHandler.getLevel1Values(rootPage);
            Set<String> level1Codes = level1Values.keySet();
            
            LevelComparator level1Comparator = new LevelComparator(level1Values);
            
            Map<String, LevelValue> level2Values = _odfPageHandler.getLevel2Values(rootPage);
            Set<String> level2Codes = level2Values.keySet();
            LevelComparator level2Comparator = new LevelComparator(level2Values);
            
            level1Tree = new TreeMap<>(level1Comparator);
            odfPageCache.put(ODFPageCacheKey.of(workspace, pageId), level1Tree);
            
            AmetysObjectIterable<Program> programs = _odfPageHandler.getProgramsWithRestrictions(rootPage, null, null, null, null);
            
            String level1MetaPath = _odfPageHandler.getLevel1Metadata(rootPage);
            String level2MetaPath = _odfPageHandler.getLevel2Metadata(rootPage);
            
            for (Program program : programs)
            {
                if (StringUtils.isBlank(level1MetaPath))
                {
                    Map<String, Collection<Program>> level2Tree = level1Tree.computeIfAbsent(__NO_FIRST_DATA_KEY, x -> new TreeMap<>(level2Comparator));
                    Collection<Program> programCache = level2Tree.computeIfAbsent(__NO_SECOND_DATA_KEY, x -> new ArrayList<>());
                    
                    programCache.add(program);
                }
                else if (StringUtils.isBlank(level2MetaPath))
                {
                    String programL1Value = _odfPageHandler.getProgramLevelValue(program, level1MetaPath);
                    if (level1Codes.contains(programL1Value))
                    {
                        Map<String, Collection<Program>> level2Tree = level1Tree.computeIfAbsent(programL1Value, x -> new TreeMap<>(level2Comparator));
                        Collection<Program> programCache = level2Tree.computeIfAbsent(__NO_SECOND_DATA_KEY, x -> new ArrayList<>());
                        
                        programCache.add(program);
                    }
                }
                else
                {
                    String programL1Value = _odfPageHandler.getProgramLevelValue(program, level1MetaPath);
                    String programL2Value = _odfPageHandler.getProgramLevelValue(program, level2MetaPath);
                    
                    if (level1Codes.contains(programL1Value) && level2Codes.contains(programL2Value))
                    {
                        Map<String, Collection<Program>> level2Tree = level1Tree.computeIfAbsent(programL1Value, x -> new TreeMap<>(level2Comparator));
                        Collection<Program> programCache = level2Tree.computeIfAbsent(programL2Value, x -> new ArrayList<>());
                        
                        programCache.add(program);
                    }
                }
            }
        }
        
        return level1Tree;
    }

    private static class LevelComparator implements Comparator<String>
    {
        private final Map<String, LevelValue> _levelValues;
        
        public LevelComparator(Map<String, LevelValue> initialMap)
        {
            _levelValues = initialMap;
        }
        
        public int compare(String s1, String s2)
        {
            if (_levelValues.containsKey(s1))
            {
                // Compare levels on orders then labels
                return _levelValues.get(s1).compareTo(_levelValues.get(s2));
            }
            else
            {
                // _no-first-data or _no-second-data
                return 0;
            }
        }
    }
    
    /**
     * Get programs to given levels
     * @param rootPage The ODF root page
     * @param level1 The value of first level or <code>null</code> if there is no first level
     * @param level2 The value of second level or <code>null</code> if there is no second level
     * @param computeIfNotPresent When false, no result will be returned if the root page in not already in the cache
     * @return The matching programs
     */
    Optional<AmetysObjectIterable<Program>> getPrograms(Page rootPage, String level1, String level2, boolean computeIfNotPresent)
    {
        return Optional.ofNullable(getProgramCache(rootPage, computeIfNotPresent))
                .map(firstLevelCache -> firstLevelCache.get(Optional.ofNullable(level1).orElse(__NO_FIRST_DATA_KEY)))
                .map(secondLevelCache -> secondLevelCache.get(Optional.ofNullable(level2).orElse(__NO_SECOND_DATA_KEY)))
                .map(CollectionIterable<Program>::new);
    }
    
    /**
     * Get the child page from its relative path.
     * @param rootPage The ODF root page
     * @param parentPage The parent page
     * @param level1 The value of first level or <code>null</code> if there is no first level
     * @param level2 The value of second level or <code>null</code> if there is no second level
     * @param path the path of the child page
     * @return the child page
     * @throws UnknownAmetysObjectException if no child page was found at the given path
     */
    Page getChildProgramPage(Page rootPage, Page parentPage, String level1, String level2, String path) throws UnknownAmetysObjectException
    {
        List<String> headQueuePath = Arrays.asList(StringUtils.split(path, "/", 2));
        String queuePath = Iterables.get(headQueuePath, 1, null); 
        
        String pageName = headQueuePath.get(0);

        ProgramPageFactory programPageFactory = (ProgramPageFactory) _ametysObjectFactoryEP.getExtension(ProgramPageFactory.class.getName());
        
        return getProgramFromPageName(rootPage, level1, level2, pageName)
            .map(program -> {
                return programPageFactory.createProgramPage(rootPage, program, null, null, parentPage);
            })
            .map(cp -> _odfPageHandler.addRedirectIfNeeded(cp, pageName))
            .map(cp -> _odfPageHandler.exploreQueuePath(cp, queuePath))
            .orElseThrow(() -> new UnknownAmetysObjectException("There's no program for page's name " + pageName));
    }
    
    /**
     * Get cached program corresponding to the page name
     * @param rootPage The ODF root page
     * @param level1 The value of first level or <code>null</code> if there is no first level
     * @param level2 The value of second level or <code>null</code> if there is no second level
     * @param pageName the page's name
     * @return The program if found
     */
    Optional<Program> getProgramFromPageName(Page rootPage, String level1, String level2, String pageName)
    {
        // Page's name is like [title]-[code]
        String programName = null;
        String programCode = null;
        
        // FIXME Search for program's name for legacy purpose. To be removed in a later version.
        // In legacy, page's name can be like the program name (program-[title])
        int j = pageName.lastIndexOf("program-");
        if (j != -1)
        {
            String name = pageName.substring(j);
            // The page name contains "program-", so it could be a legacy page name. 
            // ... but it also could be a program whose title contains the text "program"
            // ... so check if the program name exist to assure it a legacy page name
            if (_doesProgramNameExist(rootPage, name))
            {
                programName = name;
            }
        }
        
        if (programName == null)
        {
            j = pageName.lastIndexOf("-");
            programCode = j == -1 ? pageName : pageName.substring(j + 1);
        }
        
        return Optional.ofNullable(getProgram(rootPage, level1, level2, programCode, programName));
    }
    
    private boolean _doesProgramNameExist(Page rootPage, String programName)
    {
        String catalog = _odfPageHandler.getCatalog(rootPage);
        String lang = rootPage.getSitemapName();
        
        AndExpression expression = new AndExpression();
        expression.add(new ContentTypeExpression(Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE));
        expression.add(new LanguageExpression(Operator.EQ, lang));
        if (catalog != null)
        {
            expression.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        }
        
        String xPathQuery = QueryHelper.getXPathQuery(programName, "ametys:content", expression);
        AmetysObjectIterable<AmetysObject> programs = _resolver.query(xPathQuery);
        return programs.getSize() > 0;
    }
    
    /**
     * Get cached program
     * @param rootPage The ODF root page
     * @param level1 The value of first level or <code>null</code> if there is no first level
     * @param level2 The value of second level or <code>null</code> if there is no second level
     * @param programCode The code of program. Can be null if programName is not null.
     * @param programName The name of program. Can be null if programCode is not null.
     * @return program The program
     */
    Program getProgram(Page rootPage, String level1, String level2, String programCode, String programName)
    {
        String workspace = _workspaceSelector.getWorkspace();
        String pageId = rootPage.getId();
        
        // Map<level1, Map<level2, Map<program name, Program>>>
        Cache<ODFPageProgramCacheKey, Program> odfPageCache = _getODFPageProgramCache();
        
        String level1Key = StringUtils.defaultString(level1, __NO_FIRST_DATA_KEY);
        String level2Key = StringUtils.defaultString(level2, __NO_SECOND_DATA_KEY);
        
        // For legacy purpose we use the programName when the programCode is null.
        String programKey = StringUtils.defaultString(programCode, programName);
        
        return odfPageCache.get(
            ODFPageProgramCacheKey.of(workspace, pageId, level1Key, level2Key, programKey),
            key -> this._getProgramWithRestrictions(rootPage, level1, level2, programCode, programName)
        );
        
    }
    
    private Program _getProgramWithRestrictions(Page rootPage, String level1, String level2, String programCode, String programName)
    {
        return _odfPageHandler.getProgramsWithRestrictions(rootPage, level1, level2, programCode, programName).stream()
                .findFirst().orElse(null);
        
    }
    
    /**
     * Clear page cache
     * @param rootPage The ODF root page
     */
    public void clearCache (Page rootPage)
    {
        _getODFPageCache().invalidate(ODFPageCacheKey.of(null, rootPage.getId()));
        _getODFPageProgramCache().invalidate(ODFPageProgramCacheKey.of(null, rootPage.getId(), null, null, null));
    }
    
    private Cache<ODFPageCacheKey, Map<String, Map<String, Collection<Program>>>> _getODFPageCache()
    {
        return _cacheManager.get(__TREE_CACHE);
    }
    
    private static class ODFPageCacheKey extends AbstractCacheKey
    {
        public ODFPageCacheKey(String workspace, String pageId)
        {
            super(workspace, pageId);
        }
        
        public static ODFPageCacheKey of(String workspace, String pageId)
        {
            return new ODFPageCacheKey(workspace, pageId);
        }
    }
    
    private Cache<ODFPageProgramCacheKey, Program> _getODFPageProgramCache()
    {
        return _cacheManager.get(__PROGRAM_CACHE);
    }
    
    private static class ODFPageProgramCacheKey extends AbstractCacheKey
    {
        public ODFPageProgramCacheKey(String workspace, String pageId, String level1Value, String level2Value, String programName)
        {
            super(workspace, pageId, level1Value, level2Value, programName);
        }
        
        public static ODFPageProgramCacheKey of(String workspace, String pageId, String level1Value, String level2Value, String programName)
        {
            return new ODFPageProgramCacheKey(workspace, pageId, level1Value, level2Value, programName);
        }
    }
}
