/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.restrictions;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * Enumerator for available {@link OdfProgramRestriction}s
 */
public class OrgProgramRestrictionEnumerator implements Enumerator<String>, Serviceable, Configurable
{
    /** Odf program restriction manager */
    protected OdfProgramRestrictionManager _odfProgramRestrictionManager;
    
    /** True if a value must be selected, false to allow the empty value. */
    protected boolean _mandatory;
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        Configuration mandatoryConf = configuration.getChild("enumeration").getChild("custom-enumerator").getChild("mandatory", false);
        _mandatory = mandatoryConf != null ? mandatoryConf.getValueAsBoolean(true) : false;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfProgramRestrictionManager = (OdfProgramRestrictionManager) smanager.lookup(OdfProgramRestrictionManager.ROLE);
    }
    
    @Override
    public Map<String, I18nizableText> getTypedEntries() throws Exception
    {
        Map<String, I18nizableText> entries = new HashMap<>();
        
        if (!_mandatory)
        {
            entries.put(StringUtils.EMPTY, new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_NO_RESTRICTION"));
        }
        
        for (OdfProgramRestriction restriction : _odfProgramRestrictionManager.getRestrictions().values())
        {
            entries.put(restriction.getId(), restriction.getLabel());
        }
        
        return entries;
    }
    
    @Override
    public I18nizableText getEntry(String value) throws Exception
    {
        I18nizableText entry = null;
        
        if (StringUtils.isEmpty(value))
        {
            entry = new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_NO_RESTRICTION");
        }
        else
        {
            OdfProgramRestriction restriction = _odfProgramRestrictionManager.getRestrictions().get(value);
            if (restriction != null)
            {
                entry = restriction.getLabel();
            }
        }
        
        return entry;
    }
}
