/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.rights;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cache.Cache;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Implementation of {@link ODFRoleAccessControllerHelper} that consider current site to get program items with user as role.
 *
 */
public class ODFRoleAccessControllerHelper extends org.ametys.odf.rights.ODFRoleAccessControllerHelper implements Contextualizable
{
    private static final String __CACHE_BY_SITE_ID = ODFRoleAccessControllerHelper.class.getName() + "$cacheBySite";

    private SiteManager _siteManager;
    private OdfPageHandler _odfPageHandler;
    private OdfPageResolver _odfPageResolver;
    private ODFHelper _odfHelper;

    private Context _context;

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _odfPageHandler = (OdfPageHandler) smanager.lookup(OdfPageHandler.ROLE);
        _odfPageResolver = (OdfPageResolver) smanager.lookup(OdfPageResolver.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        if (!_cacheManager.hasCache(__CACHE_BY_SITE_ID))
        {
            _cacheManager.createRequestCache(__CACHE_BY_SITE_ID,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ROLE_ACCESS_CONTROLLER_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ROLE_ACCESS_CONTROLLER_DESCRIPTION"),
                false
            );
        }
    }
    
    static class CacheKey extends AbstractCacheKey
    {
        CacheKey(UserIdentity userIdentity, String siteName, String roleAttributePath)
        {
            super(userIdentity, siteName, roleAttributePath);
        }

        static CacheKey of(UserIdentity userIdentity, String siteName, String roleAttributePath)
        {
            return new CacheKey(userIdentity, siteName, roleAttributePath);
        } 
        
    }
    
    @Override
    public boolean hasODFRoleOnAnyProgramItem(UserIdentity user, String attributePath)
    {
        Cache<CacheKey, Boolean> cache = _getCacheBySite();
        
        Request request = ContextHelper.getRequest(_context);
        String siteName = WebHelper.getSiteName(request);
        
        CacheKey key = CacheKey.of(user, siteName, attributePath);
        return cache.get(key, __ -> _hasODFRoleOnAnyProgramItem(user, siteName, attributePath));
    }
    
    private boolean _hasODFRoleOnAnyProgramItem(UserIdentity user, String siteName, String attributePath)
    {
        AmetysObjectIterable<ProgramItem> programItems = _odfRightHelper.getProgramItemsWithUserAsRole(user, attributePath);
        if (programItems.getSize() == 0)
        {
            // user has no ODF role on program items
            return false;
        }
        
        if (StringUtils.isNotBlank(siteName))
        {
            Site site = _siteManager.getSite(siteName);
            
            String odfMainSite = Config.getInstance().getValue("odf.web.site.name");
            if (siteName.equals(odfMainSite))
            {
                // Allow access to main ODF site for user with a ODF role
                return true;
            }
            else if (_odfPageHandler.hasOdfRootPage(site))
            {
                // Check if at least one program item is part of ODF sitemap tree of current site (taking into account ODF restrictions)
                for (ProgramItem programItem : programItems)
                {
                    if (programItem instanceof Container)
                    {
                        for (AbstractProgram abstractProgram : _odfHelper.getParentAbstractPrograms(programItem))
                        {
                            if (!_odfPageResolver.getReferencingPages(abstractProgram, siteName, null).isEmpty())
                            {
                                return true;
                            }
                        }
                    }
                    else if (!_odfPageResolver.getReferencingPages(programItem, siteName, null).isEmpty())
                    {
                        return true;
                    }
                }
                
                // No find program item which is part of ODF tree on current site (there is no ODF root page, or ODF restrictions do not match program items)
                return false;
            }
            else
            {
                // User has ONLY access to main ODF site
                return false;
            }
        }
        else
        {
            // user has ODF role on program items on no site context
            return true;
        }
    }
    
    private Cache<CacheKey, Boolean> _getCacheBySite()
    {
        return _cacheManager.get(__CACHE_BY_SITE_ID);
    }
}
