/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.web.frontoffice.search.metamodel.FacetDefinition;
import org.ametys.web.frontoffice.search.metamodel.SearchCriterionDefinition;
import org.ametys.web.frontoffice.search.metamodel.impl.WordingSearchCriterionDefinition;

/**
 * A helper for ODF search
 */
public class ODFSearchHelper implements Serviceable, Component
{
    /** The avalon role. */
    public static final String ROLE = ODFSearchHelper.class.getName();

    /** The content type id for abstract program */
    private static final String __ABSTRACT_PROGRAM_CONTENT_TYPE_ID = "org.ametys.plugins.odf.Content.abstractProgram";
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    /**
     * Determines if the criterion matches with program and subprogram
     * @param criterion the criterion
     * @return <code>true</code> if the criterion match with program and subprogram
     */
    public boolean isCriterionOnBothProgramAndSubProgram(SearchCriterionDefinition criterion)
    {   
        ContentType abstractProgramType = _cTypeEP.getExtension(__ABSTRACT_PROGRAM_CONTENT_TYPE_ID);
        String fieldPath = getPathFromSearchFieldId(criterion.getId());
        
        return criterion instanceof WordingSearchCriterionDefinition
               || abstractProgramType.hasModelItem(fieldPath);
    }
    
    /**
     * True if the facet match with program and subprogram
     * @param facet the criterion
     * @return <code>true</code> if the facet match with program and subprogram
     */
    public boolean isFacetOnBothProgramAndSubProgram(FacetDefinition facet)
    {   
        ContentType abstractProgramType = _cTypeEP.getExtension(__ABSTRACT_PROGRAM_CONTENT_TYPE_ID);
        String fieldPath = getPathFromSearchFieldId(facet.getId());

        return abstractProgramType.hasModelItem(fieldPath);
    }
    
    /**
     * Get the path of the search field
     * @param fieldId the field id
     * @return the path of the search field
     */
    public String getPathFromSearchFieldId(String fieldId)
    {
        return StringUtils.substringAfterLast(fieldId, "$");
    }
}
