/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import java.util.List;
import java.util.Locale;
import java.util.Optional;

import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.odfweb.repository.ProgramPage;
import org.ametys.plugins.odfweb.service.search.ProgramReturnable.DisplaySubprogramMode;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.runtime.model.View;
import org.ametys.web.frontoffice.search.metamodel.ReturnableSaxer;
import org.ametys.web.frontoffice.search.metamodel.impl.PageReturnable;
import org.ametys.web.frontoffice.search.metamodel.impl.PageSaxer;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * {@link ReturnableSaxer} for {@link ProgramReturnable}
 */
public class ProgramSaxer extends PageSaxer
{
    private ProgramReturnable _programReturnable;
    private DisplaySubprogramMode _displaySubprogramMode;

    /**
     * Constructor
     * @param pageReturnable The page returnable (needed for superclass)
     * @param programReturnable The associated returnable on programs
     * @param displaySubprogramMode The mode for displaying (or not) subprograms
     */
    public ProgramSaxer(PageReturnable pageReturnable, ProgramReturnable programReturnable, DisplaySubprogramMode displaySubprogramMode)
    {
        super(pageReturnable);
        _programReturnable = programReturnable;
        _displaySubprogramMode = displaySubprogramMode;
    }
    
    @Override
    public boolean canSax(AmetysObject hit, Logger logger, SearchComponentArguments args)
    {
        return hit instanceof Program;
    }
    
    @Override
    public void sax(ContentHandler contentHandler, AmetysObject hit, Logger logger, SearchComponentArguments args) throws SAXException
    {
        Request request = ContextHelper.getRequest(_programReturnable._avalonContext);
        Program program = (Program) hit;
        Optional<ProgramPage> programPage = _resolveProgramPage(program, args.currentSite());
        if (programPage.isPresent())
        {
            _saxProgramPage(contentHandler, programPage.get(), logger, args);
        }
        else
        {
            logger.warn("The program {} was returned from the search but its ProgramPage could not be resolved", program);
            // sax minimal data for the program
            String sitemapName = args.currentPage().getSitemapName();
            _saxProgramWithNoPage(contentHandler, program, sitemapName, logger);
        }
        saxSubPrograms(contentHandler, program, programPage, logger, request);
    }
    
    private Optional<ProgramPage> _resolveProgramPage(Program program, Site currentSite)
    {
        return Optional.of(_programReturnable._getOdfPageResolver())
                .map(res -> res.getProgramPage(program, currentSite.getName()));
    }
    
    private Optional<Page> _resolveSubProgramPage(Program program, Optional<ProgramPage> programPage, SubProgram subProgram)
    {
        return programPage
                .map(ProgramPage::getSiteName)
                .map(siteName -> _programReturnable._getOdfPageResolver().getSubProgramPage(subProgram, program, siteName));
    }
    
    private String _getSubProgramPagePath(Page subProgramPage, String programPath)
    {
        return StringUtils.substringAfterLast(subProgramPage.getPathInSitemap(), programPath);
    }
    
    private void _saxProgramWithNoPage(ContentHandler contentHandler, Program program, String sitemapName, Logger logger) throws SAXException
    {
        Locale locale = new Locale(sitemapName);
        XMLUtils.createElement(contentHandler, "title", program.getTitle());
        saxContent(program, "index", locale, contentHandler, logger);
    }
    
    private void _saxProgramPage(ContentHandler contentHandler, ProgramPage programPage, Logger logger, SearchComponentArguments args) throws SAXException
    {
        super.sax(contentHandler, programPage, logger, args);
    }
    
    /**
     * SAX the subprograms of the program
     * @param contentHandler The content handler
     * @param program The program
     * @param programPage The program page
     * @param logger A logger
     * @param request The request
     * @throws SAXException if a SAX error occured
     */
    protected void saxSubPrograms(ContentHandler contentHandler, Program program, Optional<ProgramPage> programPage, Logger logger, Request request) throws SAXException
    {
        if (_displaySubprogramMode == DisplaySubprogramMode.NONE)
        {
            return;
        }

        List<String> matchingSubProgramIds = MatchingSubprogramSearchComponent._getMatchingSubProgramIds(request);
        
        ContentType subProgramCType = _programReturnable._getContentTypeEP().getExtension(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE);
        View view = subProgramCType.getView("index");
        Optional<String> programPath = programPage
                .map(ProgramPage::getPathInSitemap);
        
        for (ProgramPart childProgramPart : program.getProgramPartChildren())
        {
            if (childProgramPart instanceof SubProgram)
            {
                SubProgram subProgram = (SubProgram) childProgramPart;
                boolean matchSearch = matchingSubProgramIds.contains(subProgram.getId());
                saxSubProgram(contentHandler, subProgram, program, programPage, matchSearch, programPath, logger, view);
            }
        }
    }
    
    /**
     * SAX the subprogram
     * @param contentHandler The content handler
     * @param subProgram The subProgram
     * @param program The program
     * @param programPage The program page
     * @param matchSearch <code>true</code> if the subprogram matches the search
     * @param programPath The program path
     * @param logger A logger
     * @param view The view
     * @throws SAXException if a SAX error occured
     */
    protected void saxSubProgram(
            ContentHandler contentHandler, 
            SubProgram subProgram, 
            Program program, 
            Optional<ProgramPage> programPage, 
            boolean matchSearch, 
            Optional<String> programPath, 
            Logger logger, 
            View view) throws SAXException
    {
        if (_displaySubprogramMode != DisplaySubprogramMode.MATCHING_SEARCH_ONLY || matchSearch)
        {
            AttributesImpl attrs = new AttributesImpl();
            Optional<Page> subProgramPage = _resolveSubProgramPage(program, programPage, subProgram);
            if (subProgramPage.isPresent() && programPath.isPresent())
            {
                attrs.addCDATAAttribute("path", _getSubProgramPagePath(subProgramPage.get(), programPath.get()));
            }
            attrs.addCDATAAttribute("id", subProgram.getId());
            attrs.addCDATAAttribute("name", subProgram.getName());
            attrs.addCDATAAttribute("title", subProgram.getTitle());
            if (_displaySubprogramMode == DisplaySubprogramMode.ALL_WITH_HIGHLIGHT)
            {
                attrs.addCDATAAttribute("highlight", String.valueOf(matchSearch));
            }
            XMLUtils.startElement(contentHandler, "subprogram", attrs);

            try
            {
                subProgram.dataToSAX(contentHandler, view);
            }
            catch (Exception e)
            {
                logger.error("An error occurred during saxing subprogram '" + subProgram.getId() + "' metadata", e);
            }
            
            XMLUtils.endElement(contentHandler, "subprogram");
        }
    }
}
