/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search.helper;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.search.ui.model.SearchUICriterion;
import org.ametys.cms.search.ui.model.impl.IndexingFieldSearchUICriterion;
import org.ametys.core.util.I18nUtils;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.odfweb.service.search.criterion.DegreeUniversityAttributeContentSearchCriterionDefinition;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.parameter.Validator;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugin.component.ThreadSafeComponentManager;
import org.ametys.web.frontoffice.search.metamodel.Searchable;

/**
 * The helper for degree university
 */
public class DegreeUniversityHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable, PluginAware
{
    /** The avalon role. */
    public static final String ROLE = DegreeUniversityHelper.class.getName();

    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    /** The service manager */
    protected ServiceManager _manager;
    
    /** The context */
    protected Context _context;
    
    /** The plugin name */
    protected String _pluginName;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _contentHelper = (ContentHelper) manager.lookup(ContentHelper.ROLE);
        _manager = manager;
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    /**
     * Get all degree contents of type university or not
     * @param isUniversity <code>true</code> to return all degree contents of type university
     * @return the list of degree contents
     */
    public List<Content> getDegrees(boolean isUniversity)
    {
        StringExpression duExpression = new StringExpression(DegreeUniversityAttributeContentSearchCriterionDefinition.ATTRIBUTE_DEGREE_TYPE, Operator.EQ, DegreeUniversityAttributeContentSearchCriterionDefinition.ATTRIBUTE_DEGREE_TYPE_UNIVERSITY_VALUE);
        Expression expr = new AndExpression(
            new ContentTypeExpression(Operator.EQ, OdfReferenceTableHelper.DEGREE),
            isUniversity ? duExpression : new NotExpression(duExpression)
        );
        
        return _resolver.query(QueryHelper.getXPathQuery(null, RepositoryConstants.NAMESPACE_PREFIX + ":content", expr))
            .stream()
            .filter(Content.class::isInstance)
            .map(Content.class::cast)
            .collect(Collectors.toList());
    }
    
    /**
     * Get the degree university search criterion definition
     * @param searcheable the searcheable link to the criterion definition
     * @return the search criterion definition
     */
    public DegreeUniversityAttributeContentSearchCriterionDefinition getDegreeUniversityCriterionDefinition(Searchable searcheable)
    {
        try
        {
            ThreadSafeComponentManager<SearchUICriterion> searchCriterionManager = new ThreadSafeComponentManager<>();
            searchCriterionManager.setLogger(getLogger());
            searchCriterionManager.contextualize(_context);
            searchCriterionManager.service(_manager);
            
            Configuration criteriaConf = _getDegreeIndexingFieldCriteriaConfiguration();
            searchCriterionManager.addComponent(_pluginName, null, AbstractProgram.DEGREE, IndexingFieldSearchUICriterion.class, criteriaConf);
            
            searchCriterionManager.initialize();
            
            ContentType programContentType = _cTypeEP.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
            
            Optional<Validator> validator = _getDegreeValidator(programContentType);
            SearchUICriterion criterion = searchCriterionManager.lookup(AbstractProgram.DEGREE);
            
            return new DegreeUniversityAttributeContentSearchCriterionDefinition(
                _pluginName, 
                Optional.of(searcheable),
                criterion, 
                Optional.of(programContentType), 
                validator, 
                _resolver, 
                _cTypeEP,
                _contentHelper,
                _i18nUtils,
                this
            );
        }
        catch (Exception e)
        {
            throw new RuntimeException("An error occured when retrieving IndexingFieldSearchCriterionDefinitions", e);
        }
    }
    
    private Optional<Validator> _getDegreeValidator(ContentType contentType) 
    {
        return Optional.of(contentType.getModelItem(AbstractProgram.DEGREE))
                .filter(ElementDefinition.class::isInstance)
                .map(ElementDefinition.class::cast)
                .map(ElementDefinition::getValidator);
    }
    
    private Configuration _getDegreeIndexingFieldCriteriaConfiguration()
    {
        DefaultConfiguration criteriaConf = new DefaultConfiguration("criteria");
        DefaultConfiguration metaConf = new DefaultConfiguration("field");
        criteriaConf.addChild(metaConf);
        metaConf.setAttribute("path", AbstractProgram.DEGREE);
        
        DefaultConfiguration cTypesConf = new DefaultConfiguration("contentTypes");
        criteriaConf.addChild(cTypesConf);
        DefaultConfiguration baseTypeConf = new DefaultConfiguration("baseType");
        baseTypeConf.setAttribute("id", ProgramFactory.PROGRAM_CONTENT_TYPE);
        cTypesConf.addChild(baseTypeConf);
        
        return criteriaConf;
    }
}
