/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;

/**
 * Resolves an ODF page path from the associated ODF content.
 */
public class OdfPageResolver extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The avalon role. */
    public static final String ROLE = OdfPageResolver.class.getName();
    
    private static final String __PATH_IN_SITEMAP_CACHE = OdfPageResolver.class.getName() + "$pathInSitemap";
    private static final String __LEVELS_PATH_CACHE = OdfPageResolver.class.getName() + "$levelsPath";
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;
    /** The odf page handler */
    protected OdfPageHandler _odfPageHandler;
    /** ODF helper */
    protected ODFHelper _odfHelper;
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _ametysResolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _odfPageHandler = (OdfPageHandler) serviceManager.lookup(OdfPageHandler.ROLE);
        _odfHelper = (ODFHelper) serviceManager.lookup(ODFHelper.ROLE);
        _cacheManager = (AbstractCacheManager) serviceManager.lookup(AbstractCacheManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createRequestCache(__PATH_IN_SITEMAP_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_PATH_IN_SITEMAP_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_PATH_IN_SITEMAP_DESCRIPTION"),
                false);
        _cacheManager.createRequestCache(__LEVELS_PATH_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_LEVELS_PATH_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_LEVELS_PATH_DESCRIPTION"),
                false);
    }
    
    /**
     * Get all referencing pages for this program item, in all sites and all sitemaps
     * @param programItem The program item
     * @return the referencing pages
     */
    public Set<Page> getReferencingPages(ProgramItem programItem)
    {
        return getReferencingPages(programItem, null, ((Content) programItem).getLanguage());
    }
    
    /**
     * Get all referencing pages for this program item
     * @param programItem The program item
     * @param siteName The site name. Can be null to search on all sites
     * @param lang The sitemap language. Can be null to search on all sitemaps
     * @return the referencing pages
     */
    public Set<Page> getReferencingPages(ProgramItem programItem, String siteName, String lang)
    {
        Set<Page> refPages = new HashSet<>();
        
        Set<Page> odfRootPages = _odfPageHandler.getOdfRootPages(siteName, lang);
        
        for (Page rootPage : odfRootPages)
        {
            if (programItem instanceof Program)
            {
                ProgramPage programPage = getProgramPage(rootPage, (Program) programItem);
                if (programPage != null)
                {
                    refPages.add(programPage);
                }
            }
            else if (programItem instanceof SubProgram)
            {
                Set<Program> parentPrograms = _odfHelper.getParentPrograms(programItem);
                for (Program parentProgram : parentPrograms)
                {
                    ProgramPage subProgramPage = getSubProgramPage(rootPage, (SubProgram) programItem, parentProgram);
                    if (subProgramPage != null)
                    {
                        refPages.add(subProgramPage);
                    }
                }
            }
            else if (programItem instanceof Course)
            {
                List<CourseList> parentCourseLists = ((Course) programItem).getParentCourseLists();
                for (CourseList courseList : parentCourseLists)
                {
                    List<Course> parentCourses = courseList.getParentCourses();
                    for (Course parentCourse : parentCourses)
                    {
                        CoursePage coursePage = getCoursePage(rootPage, (Course) programItem, parentCourse);
                        if (coursePage != null)
                        {
                            refPages.add(coursePage);
                        }
                    }
                    
                    List<AbstractProgram> parentAbstractPrograms = getNearestAncestorAbstractPrograms(courseList);
                    for (AbstractProgram parentAbstractProgram : parentAbstractPrograms)
                    {
                        CoursePage coursePage = getCoursePage(rootPage, (Course) programItem, parentAbstractProgram);
                        if (coursePage != null)
                        {
                            refPages.add(coursePage);
                        }
                    }
                }
            }
        }
        
        return refPages;
    }
    
    /**
     * Return the program page
     * @param program the program
     * @return the page program or null
     */
    public ProgramPage getProgramPage(Program program)
    {
        return getProgramPage(program, null);
    }
    
    /**
     * Return the program page
     * @param program the program
     * @param siteName The current site name. If the no ODF root page is present in this site, the default ODF site will be used instead.
     * @return the page program or null
     */
    public ProgramPage getProgramPage(Program program, String siteName)
    {
        Page odfRootPage = getOdfRootPage(siteName, program.getLanguage(), program.getCatalog());
        
        if (odfRootPage == null)
        {
            return null;
        }
        
        return getProgramPage(odfRootPage, program);
    }
    
    /**
     * Return the program page
     * @param odfRootPage the odf root page
     * @param program the program
     * @return the page program or null
     */
    public ProgramPage getProgramPage (Page odfRootPage, Program program)
    {
        // E.g: program://_root?rootId=xxxx&programId=xxxx
        String pageId =  "program://_root?rootId=" + odfRootPage.getId() + "&programId=" + program.getId();
        try
        {
            return _ametysResolver.resolveById(pageId);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Return the subprogram page
     * @param subProgram the subprogram
     * @param parentProgram The parent program
     * @param siteName The current site name. If the no ODF root page is present in this site, the default ODF site will be used instead.
     * @return the subprogram page or null
     */
    public ProgramPage getSubProgramPage(SubProgram subProgram, AbstractProgram parentProgram, String siteName)
    {
        Page odfRootPage = getOdfRootPage(siteName, subProgram.getLanguage(), subProgram.getCatalog());
        
        if (odfRootPage == null)
        {
            return null;
        }
        
        return getSubProgramPage(odfRootPage, subProgram, parentProgram);
    }
    
    /**
     * Return the subprogram page
     * @param odfRootPage the odf root page
     * @param subProgram the subprogram
     * @param parentAbstractProgram The parent program or subprogram
     * @return the subprogram page or null
     */
    public ProgramPage getSubProgramPage (Page odfRootPage, SubProgram subProgram, AbstractProgram parentAbstractProgram)
    {
        AbstractProgram parent = getNearestAncestorAbstractProgram(subProgram, parentAbstractProgram);
        Program parentProgram = getParentProgram(subProgram, parentAbstractProgram);
        
        if (parent == null || parentProgram == null)
        {
            // No page
            return null;
        }
     
        // Id is like program://path/to/subprogram?rootId=xxxx&programId=xxxx&parentId=xxxx
        String pageId =  "program://" + getPathInProgram(parent, parentProgram) + "?rootId=" + odfRootPage.getId() + "&programId=" + subProgram.getId() + "&parentId=" + parentProgram.getId();
        try
        {
            return _ametysResolver.resolveById(pageId);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Return the course page
     * @param course the course
     * @param parentProgram the parent program or subprogram. Can be null.
     * @param siteName The current site name. If the no ODF root page is present in this site, the default ODF site will be used instead.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage(Course course, AbstractProgram parentProgram, String siteName)
    {
        String catalog = course.getCatalog();
        Page odfRootPage = getOdfRootPage(siteName, course.getLanguage(), catalog);
        
        if (odfRootPage == null)
        {
            return null;
        }
        
        return getCoursePage(odfRootPage, course, parentProgram);
    }
    
    /**
     * Return the course page
     * @param odfRootPage the odf root page
     * @param course the course
     * @param parentAbstractProgram the parent program or subprogram. Can be null.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage (Page odfRootPage, Course course, AbstractProgram parentAbstractProgram)
    {
        ProgramItem parent = null;
        
        Course parentCourse = getNearestAncestorCourse(course, parentAbstractProgram);
        if (parentCourse != null)
        {
            parent = parentCourse;
        }
        else
        {
            parent = getNearestAncestorAbstractProgram(course, parentAbstractProgram);
        }
        Program parentProgram = getParentProgram(course, parentAbstractProgram);
        
        if (parent == null || parentProgram == null)
        {
            // No page
            return null;
        }
     
        // Test program restriction
        if (!_odfPageHandler.isValidRestriction(odfRootPage, parentProgram))
        {
            return null;
        }
        
        // Id is like course://path/from/program?rootId=xxx&courseId=xxx&programId=xxxx
        String pageId =  "course://" + getPathInProgram(parent, parentProgram) + "?rootId=" + odfRootPage.getId() + "&courseId=" + course.getId() + "&programId=" + parentProgram.getId();
        try
        {
            return _ametysResolver.resolveById(pageId);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Return the course page
     * @param course the course
     * @param parentCourse the parent course. Can NOT be null.
     * @param siteName The current site name. If the no ODF root page is present in this site, the default ODF site will be used instead.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage (Course course, Course parentCourse, String siteName)
    {
        String catalog = course.getCatalog();
        Page odfRootPage = getOdfRootPage(siteName, course.getLanguage(), catalog);
        
        if (odfRootPage == null)
        {
            return null;
        }
        
        return getCoursePage(odfRootPage, course, parentCourse);
    }
    
    /**
     * Return the course page
     * @param odfRootPage the odf root page
     * @param course the course
     * @param parentCourse the parent course. Can NOT be null.
     * @return the course page or null if not found
     */
    public CoursePage getCoursePage (Page odfRootPage, Course course, Course parentCourse)
    {
        AbstractProgram parent = getNearestAncestorAbstractProgram(parentCourse, null);
        Program parentProgram = getParentProgram(parentCourse, null);
        
        if (parent == null || parentProgram == null)
        {
            // No page
            return null;
        }
     
        // Test program restriction
        if (!_odfPageHandler.isValidRestriction(odfRootPage, parentProgram))
        {
            return null;
        }
        
        // Id is like course://path/from/program?rootId=xxx&courseId=xxx&programId=xxxx
        String pageId =  "course://" + getPathInProgram(parentCourse, parentProgram) + "?rootId=" + odfRootPage.getId() + "&courseId=" + course.getId() + "&programId=" + parentProgram.getId();
        try
        {
            return _ametysResolver.resolveById(pageId);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Get the ODF root page, either in the given site if it exists, or in the default ODF site.
     * @param siteName the desired site name.
     * @param language the sitemap language to search in.
     * @param catalog The ODF catalog
     * @return the ODF root page, either in the given site if it exists, or in the default ODF site.
     */
    public Page getOdfRootPage(String siteName, String language, String catalog)
    {
        Page odfRootPage = null;
        
        if (StringUtils.isNotEmpty(siteName))
        {
            odfRootPage = _odfPageHandler.getOdfRootPage(siteName, language, catalog);
        }
        
        if (odfRootPage == null)
        {
            String odfSiteName = Config.getInstance().getValue("odf.web.site.name");
            odfRootPage = _odfPageHandler.getOdfRootPage(odfSiteName, language, catalog);
        }
        
        return odfRootPage;
    }

    /**
     * Get the path in sitemap of a ODF content into a {@link Program} or {@link SubProgram}<br>
     * Be careful, this is the path in sitemap, to get the path of a item into a Program, use {@link ODFHelper#getPathInProgram} instead.
     * @param programItem The program item
     * @param parentProgram The parent root (sub)program. Can be null.
     * @return the path in sitemap from the parent program or null if no found ODF path for those program item and parent program
     */
    public String getPathInProgram (ProgramItem programItem, AbstractProgram parentProgram)
    {
        Cache<PathInProgramCacheKey, String> cache = _cacheManager.get(__PATH_IN_SITEMAP_CACHE);
        
        return cache.get(PathInProgramCacheKey.of(programItem.getId(), parentProgram != null ? parentProgram.getId() : "__NOPARENT"), item -> {
            
            if (programItem instanceof Program || programItem.equals(parentProgram))
            {
                // The program item is already the program it self
                return _odfPageHandler.getPageName(programItem);
            }
            
            List<String> paths = new ArrayList<>();
            
            // Add the parent path in program if exists
            ProgramItem parent = _odfHelper.getParentProgramItem(programItem, parentProgram);
            if (parent != null)
            {
                String parentPath = getPathInProgram(parent, parentProgram);
                if (parentPath != null)
                {
                    paths.add(parentPath);
                }
            }
            
            // Add the current page name (if it is an item with a page (only course, subprogram and program)
            if (programItem instanceof AbstractProgram || programItem instanceof Course)
            {
                paths.add(_odfPageHandler.getPageName(programItem));
            }
            
            // If the path is empty, return null
            return paths.isEmpty() ? null : StringUtils.join(paths, "/");
        });
    }
    
    private static class PathInProgramCacheKey extends AbstractCacheKey
    {
        protected PathInProgramCacheKey(String programItemId, String parentProgramId)
        {
            super(programItemId, parentProgramId);
        }
        
        public static PathInProgramCacheKey of(String programItemId, String parentProgramId)
        {
            return new PathInProgramCacheKey(programItemId, parentProgramId);
        }
    }
    
    /**
     * Returns the first {@link Program} ancestor, ensuring that the given parent content 'parentProgram' is in the hierarchy, if not null.<br>
     * If 'parentProgram' is null, the first {@link Program} ancestor will be returned regardless of parent hierarchy.<br>
     * If 'parentProgram' is a {@link SubProgram}, the first {@link Program} ancestor from this {@link SubProgram} will be returned regardless of parent hierarchy
     * @param programItem a {@link ProgramItem}
     * @param parentProgram The parent program or subprogram. Can be null.
     * @return the parent {@link Program} into this (sub)program or null if not found
     */
    public Program getParentProgram (ProgramItem programItem, AbstractProgram parentProgram)
    {
        AbstractProgram parent = parentProgram;
        
        ProgramItem parentItem = _odfHelper.getParentProgramItem(programItem, parentProgram);
        while (parentItem != null && !(parentItem instanceof Program))
        {
            if (parent != null && parentItem.equals(parent))
            {
                // Once the desired abstract program parent is passed, the parent is null
                parent = null;
            }
            parentItem = _odfHelper.getParentProgramItem(parentItem, parent);
        }
        
        return parentItem != null ? (Program) parentItem : null;
    }
    
    /**
     * Returns the nearest {@link AbstractProgram} ancestors.
     * @param programPart a {@link ProgramPart}
     * @return the nearest {@link AbstractProgram} ancestors containing this program part
     */
    public List<AbstractProgram> getNearestAncestorAbstractPrograms (ProgramPart programPart)
    {
        List<AbstractProgram> ancestors = new ArrayList<>();
        
        List<ProgramPart> parents = programPart.getProgramPartParents();
        for (ProgramPart parent : parents)
        {
            if (parent instanceof AbstractProgram)
            {
                ancestors.add((AbstractProgram) parent);
            }
            else
            {
                ancestors.addAll(getNearestAncestorAbstractPrograms(parent));
            }
        }
        
        return ancestors;
    }
    
    /**
     * Returns the nearest {@link AbstractProgram} ancestor.
     * @param programItem a {@link ProgramItem}
     * @param parentProgram The parent program or subprogram
     * @return the nearest {@link AbstractProgram} ancestor into this (sub)program or null if not found
     */
    public AbstractProgram getNearestAncestorAbstractProgram (ProgramItem programItem, AbstractProgram parentProgram)
    {
        ProgramItem parentItem = _odfHelper.getParentProgramItem(programItem, parentProgram);
        while (parentItem != null && !(parentItem instanceof AbstractProgram))
        {
            parentItem = _odfHelper.getParentProgramItem(parentItem, parentProgram);
        }
        
        return parentItem != null ? (AbstractProgram) parentItem : null;
    }
    
    /**
     * Returns the nearest {@link Course} ancestor.
     * @param course a {@link Course}
     * @param parentProgram The parent program or subprogram
     * @return the nearest {@link Course} ancestor into this (sub)program or null if not found
     */
    public Course getNearestAncestorCourse (Course course, AbstractProgram parentProgram)
    {
        ProgramItem parentItem = _odfHelper.getParentProgramItem(course, parentProgram);
        while (parentItem != null && !(parentItem instanceof Course) && !(parentItem instanceof AbstractProgram))
        {
            parentItem = _odfHelper.getParentProgramItem(parentItem, parentProgram);
        }
        
        return parentItem != null && parentItem instanceof Course ? (Course) parentItem : null;
    }
    
    /**
     * Get the path of a {@link ProgramItem} page into the given {@link Program}
     * @param siteName the site name
     * @param language the language
     * @param programItem the subprogram.
     * @param parentProgram The parent program
     * @return the page path or empty if no page matches
     */
    public String getProgramItemPagePath(String siteName, String language, ProgramItem programItem, Program parentProgram)
    {
        String catalog = programItem.getCatalog();
        
        // Get the path with the language, path to odf root page, and levels, finishing by a /
        Cache<ODFLevelsPathCacheKey, String> cache = _cacheManager.get(__LEVELS_PATH_CACHE);
        String rootPath = cache.get(ODFLevelsPathCacheKey.of(siteName, language, catalog, parentProgram.getId()), key -> _getLevelsPath(siteName, language, programItem.getCatalog(), parentProgram));
        
        if (rootPath != null)
        {
            // Add the path from the parent program to the program item
            String pathInProgram = getPathInProgram(programItem, parentProgram);
            if (pathInProgram != null)
            {
                return rootPath + pathInProgram;
            }
        }
        
        return StringUtils.EMPTY;
    }
    
    private String _getLevelsPath(String siteName, String language, String catalog, Program parentProgram)
    {
        // Get the ODF root page for given site, language and catalog
        Page rootPage = _odfPageHandler.getOdfRootPage(siteName, language, catalog);
        
        StringBuilder sb = new StringBuilder()
            // Sitemap (language)
            .append(rootPage.getSitemapName()).append('/')
            // Path from sitemap to ODF root page
            .append(rootPage.getPathInSitemap()).append('/');
        
        // Add level1 if defined
        String level1 = _odfPageHandler.getLevel1PageName(rootPage, parentProgram);
        if (StringUtils.isNotEmpty(level1))
        {
            sb.append(level1).append('/');
        }
        
        // Add level2 if defined
        String level2 = _odfPageHandler.getLevel2PageName(rootPage, parentProgram);
        if (StringUtils.isNotEmpty(level1))
        {
            sb.append(level2).append('/');
        }
        
        return sb.toString();
    }
    
    private static class ODFLevelsPathCacheKey extends AbstractCacheKey
    {
        protected ODFLevelsPathCacheKey(String site, String lang, String catalog, String parentProgramId)
        {
            super(site, lang, catalog, parentProgramId);
        }
        
        public static ODFLevelsPathCacheKey of(String site, String lang, String catalog, String parentProgramId)
        {
            return new ODFLevelsPathCacheKey(site, lang, catalog, parentProgramId);
        }
    }
}
