/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

import com.google.common.collect.Iterables;

/**
 * Page representing a {@link Program} or a {@link SubProgram}.
 */
public class ProgramPage extends AbstractProgramItemPage<ProgramPageFactory>
{
    private AbstractProgram _program;
    private String _path;
    private Page _parentPage;
    private Program _parentProgram;
    
    /**
     * Constructor for program page holding a {@link Program} or {@link SubProgram}
     * @param factory The factory
     * @param root the ODF root page.
     * @param program the program or subprogram.
     * @param path The path from the virtual second level page. Can be null if abstract program is a {@link Program}
     * @param parentProgram the parent program in case of a subprogram, null otherwise
     * @param parentPage the parent {@link Page} or null if not yet computed.
     * @param configuration The program virtual page's configuration
     */
    public ProgramPage(Page root, VirtualPageConfiguration configuration, ProgramPageFactory factory, AbstractProgram program, String path, Program parentProgram, Page parentPage)
    {
        super(root, configuration, factory.getScheme(), factory);
        
        _program = program;
        _path = path;
        _parentPage = parentPage;
        _parentProgram = parentProgram;
    }
    
    /**
     * Returns the associated {@link Program} or {@link SubProgram}.
     * @return the associated {@link Program} or {@link SubProgram}.
     */
    public AbstractProgram getProgram()
    {
        return _program;
    }
    
    @Override
    protected ProgramItem getProgramItem()
    {
        return getProgram();
    }
    
    @Override
    public int getDepth() throws AmetysRepositoryException
    {
        int levelDepth = 0;
        if (StringUtils.isNotBlank(_factory.getODFPageHandler().getLevel1Metadata(_root)))
        {
            levelDepth++;
            if (StringUtils.isNotBlank(_factory.getODFPageHandler().getLevel2Metadata(_root)))
            {
                levelDepth++;
            }
        }
        
        return _root.getDepth() + levelDepth + (_path != null ? _path.split("/").length : 0);
    }

    @Override
    public String getTitle() throws AmetysRepositoryException
    {
        return _program.getTitle();
    }

    @Override
    public String getLongTitle() throws AmetysRepositoryException
    {
        return _program.getTitle();
    }
 
    @Override
    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        Collection<Page> children = _transformChildrenPages(_traverseChildren(_program)).toList();
        return new CollectionIterable<>(children);
    }
    
    private Page _createChildPage(ProgramItem child)
    {
        if (child instanceof SubProgram subProgram)
        {
            return _createChildProgramPage(subProgram);
        }
        else if (child instanceof Course course)
        {
            return _createChildCoursePage(course);
        }
        
        return null;
    }
    
    private ProgramPage _createChildProgramPage(SubProgram child)
    {
        return _factory.createProgramPage(_root, child, _path != null ? _path + '/' + getName() : getName(), _getParentProgram(), this);
    }
    
    private CoursePage _createChildCoursePage(Course course)
    {
        return _factory.getCoursePageFactory().createCoursePage(_root, course, _getParentProgram(), _path != null ? _path + '/' + getName() : getName(), this);
    }
    
    @Override
    public String getPathInSitemap() throws AmetysRepositoryException
    {
        String path = _computePath(_root.getPathInSitemap());
        return path == null ? null : path + "/" + getName();
    }
    
    private Program _getParentProgram()
    {
        return Optional.ofNullable(_parentProgram)
            .orElseGet(() -> (Program) _program);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path.isEmpty())
        {
            throw new AmetysRepositoryException("path must be non empty");
        }
        
        List<String> headQueuePath = Arrays.asList(StringUtils.split(path, "/", 2));
        String name = headQueuePath.get(0);
        String queuePath = Iterables.get(headQueuePath, 1, null);
        
        return (A) _findChildPage(_program, name)
            .map(cp -> _factory.getODFPageHandler().addRedirectIfNeeded(cp, name))
            .map(cp -> _factory.getODFPageHandler().exploreQueuePath(cp, queuePath))
            .orElseThrow(() -> new UnknownAmetysObjectException("Unknown child page '" + path + "' for page " + getId()));
    }
    
    private Optional<Page> _findChildPage(TraversableProgramPart parent, String name)
    {
        return _transformChildrenPages(_traverseChildren(parent).filter(child -> _filterByName(child, name))).findFirst();
    }
    
    private boolean _filterByName(ProgramItem programItem, String name)
    {
        // If last part is equals to the program item code, the page matches
        if (programItem.getCode().equals(name.substring(name.lastIndexOf("-") + 1)))
        {
            return true;
        }
        
        if (programItem instanceof SubProgram subProgram)
        {
            // For legacy purpose we use the subProgramName when the subProgramCode is null.
            String subProgramPageName = NameHelper.filterName(subProgram.getTitle()) + "-" + programItem.getName();
            return name.equals(subProgramPageName);
        }
        
        return false;
    }
    
    @Override
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        return _findChildPage(_program, name).isPresent();
    }
    
    @Override
    public String getId() throws AmetysRepositoryException
    {
        // E.g: program://_root?rootId=xxxx&programId=xxxx (for a program)
        // E.g: program://path/to/subprogram?rootId=xxxx&programId=xxxx&parentId=xxxx (for a subprogram)
        StringBuilder sb = new StringBuilder("program://");
        sb.append(StringUtils.isNotEmpty(_path) ? _path : "_root");
        sb.append("?rootId=").append(_root.getId());
        sb.append("&programId=").append(_program.getId());
        
        if (_parentProgram != null)
        {
            sb.append("&parentId=").append(_parentProgram.getId());
        }
        
        return sb.toString();
    }

    @Override
    public String getName() throws AmetysRepositoryException
    {
        // E.g: licence-lea-anglais-allemand-H7AIIUYW
        return _factory.getODFPageHandler().getPageName(_program);
    }

    @SuppressWarnings("unchecked")
    @Override
    public Page getParent() throws AmetysRepositoryException
    {
        if (_parentPage == null)
        {
            String childPath = _computePath(null);
            if (childPath != null)
            {
                _parentPage = childPath.isEmpty() ? _root : _root.getChild(childPath);
            }
        }
        
        return _parentPage;
    }

    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        return _computePath(_root.getPath());
    }
    
    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }

    private Stream<ProgramItem> _traverseChildren(TraversableProgramPart parent)
    {
        Predicate<ProgramItem> isSubProgram = SubProgram.class::isInstance;
        Predicate<ProgramItem> isCourse = Course.class::isInstance;
        
        ProgramPartTraverser traverser = new ProgramPartTraverser(parent.getProgramPartChildren());
        return traverser.stream().filter(isSubProgram.or(isCourse)).distinct();
    }
    
    /**
     * Program part traverser. Iterate recursively on child program base.
     */
    static class ProgramPartTraverser extends AbstractTreeIterator<ProgramItem>
    {
        public ProgramPartTraverser(Collection<? extends ProgramItem> programPartChildren)
        {
            super(programPartChildren);
        }

        @Override
        protected Iterator<ProgramItem> provideChildIterator(ProgramItem parent)
        {
            if (parent instanceof CourseList courseList)
            {
                return new ProgramPartTraverser(courseList.getCourses());
            }
            
            if (parent instanceof Container container)
            {
                return new ProgramPartTraverser(container.getProgramPartChildren());
            }
            
            return null;
        }
    }
    
    /**
     * Breadth first search iterator for tree structure
     * Each node can provide an iterator that will be put in the end of the queue.
     * @param <T> A tree item
     */
    abstract static class AbstractTreeIterator<T> implements Iterator<T>
    {
        protected final Queue<Iterator<T>> _nodeIterators = new LinkedList<>();
        private Boolean _hasNext;
        
        AbstractTreeIterator(Iterator<T> iterator)
        {
            if (iterator != null && iterator.hasNext())
            {
                _nodeIterators.add(iterator);
            }
        }
        
        AbstractTreeIterator(Collection<? extends T> children)
        {
            this(handleConstructorChildren(children));
        }
        
        private static <T> Iterator<T> handleConstructorChildren(Collection<? extends T> children)
        {
            Collection<T> tChildren = Collections.unmodifiableCollection(children);
            return tChildren.iterator();
        }
        
        public boolean hasNext()
        {
            if (_hasNext != null)
            {
                return _hasNext;
            }
            
            Iterator<T> it = _getOrUpdateHead();
            if (_hasNext == null)
            {
                _hasNext = it != null ? it.hasNext() : false;
            }
            
            return _hasNext;
        }
        
        public T next()
        {
            if (BooleanUtils.isFalse(_hasNext))
            {
                throw new NoSuchElementException();
            }
            
            Iterator<T> it = null;
            if (_hasNext == null)
            {
                it = _getOrUpdateHead();
            }
            else
            {
                it = _nodeIterators.peek();
            }
            
            T next = Optional.ofNullable(it)
                .map(Iterator::next)
                .orElseThrow(NoSuchElementException::new);
            
            Iterator<T> childIterator = provideChildIterator(next);
            if (childIterator != null && childIterator.hasNext())
            {
                _nodeIterators.add(childIterator);
            }
            
            // reset cached has next
            _hasNext = null;
            
            return next;
        }
        
        protected abstract Iterator<T> provideChildIterator(T next);
        
        public Stream<T> stream()
        {
            Iterable<T> iterable = () -> this;
            return StreamSupport.stream(iterable.spliterator(), false);
        }
        
        private Iterator<T> _getOrUpdateHead()
        {
            return Optional.ofNullable(_nodeIterators.peek())
                .filter(it ->
                {
                    if (it.hasNext())
                    {
                        _hasNext = true;
                        return true;
                    }
                    
                    return false;
                })
                .orElseGet(() -> _updateHead());
        }
        
        private Iterator<T> _updateHead()
        {
            _nodeIterators.poll(); // remove actual head
            return _nodeIterators.peek();
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public AbstractProgram getContent()
    {
        AbstractProgram program = getProgram();
        program.setContextPath(getPathInSitemap());
        
        if (!_factory.isIndexing())
        {
            // computing educational paths is actually very expensive, and only useful during rendering
            // we are very conservative here and only disable that computing for specific indexing cases
            // (we could have chosen to only enable it when rendering, but we don't want to forget specific cases)
            setCurrentEducationalPaths(program);
        }
        
        return program;
    }
    
    private String _computePath(String rootPath)
    {
        String levelsPath = _factory.getODFPageHandler().computeLevelsPath(_root, _getParentProgram());
        
        // The current program has no valid attributes for the levels selected in the ODF root
        if (levelsPath == null)
        {
            throw new UnknownAmetysObjectException("Page of program " + _getParentProgram().getId() + " does not have a valid level path");
        }
        
        return Stream.of(rootPath, levelsPath, _path)
            .filter(StringUtils::isNotEmpty)
            .collect(Collectors.joining("/"));
    }
    
    private Stream<Page> _transformChildrenPages(Stream<ProgramItem> children)
    {
        return children
            .map(this::_createChildPage)
            .filter(Objects::nonNull)
            // Test if the child page is in existing virtual pages
            .filter(page -> {
                try
                {
                    page.getPathInSitemap();
                    return true;
                }
                catch (UnknownAmetysObjectException e)
                {
                    return false;
                }
            });
    }
}
