/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.restrictions.rules;

import java.util.Arrays;
import java.util.Collections;
import java.util.Optional;

import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.type.ElementType;

/**
 * Metadata restriction rule
 */
public class OdfMetadataRestrictionRule implements OdfRestrictionRule
{
    private final String _metadataPath;
    private final String _untypedValue;
    
    /**
     * Constructor for rule relating to the value of a metadata
     * @param metadataPath The path of metadata
     * @param untypedValue The metadata untyped value
     */
    public OdfMetadataRestrictionRule(String metadataPath, String untypedValue)
    {
        _metadataPath = metadataPath;
        _untypedValue = untypedValue;
    }
    
    @Override
    public Expression getExpression()
    {
        return new StringExpression(_metadataPath, Operator.EQ, _untypedValue);
    }
    
    @Override
    public boolean contains(Program program)
    {
        ElementType type = (ElementType) program.getDefinition(_metadataPath).getType();
        Object typedValue = type.castValue(_untypedValue);
        
        return Optional.ofNullable(program.getValue(_metadataPath))
                       .map(value -> Arrays.asList(value))
                       .orElse(Collections.EMPTY_LIST)
                       .stream()
                       .anyMatch(value -> value.equals(typedValue));
    }
    
    @Override
    public boolean hasOrgunitRestrictions()
    {
        return false;
    }
}
