/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search.criterion;

import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.search.ui.model.SearchUICriterion;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.plugins.odfweb.service.search.ProgramSearchable;
import org.ametys.plugins.odfweb.service.search.helper.DegreeUniversityHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.parameter.Validator;
import org.ametys.web.frontoffice.search.metamodel.Searchable;
import org.ametys.web.frontoffice.search.metamodel.impl.ContentAttributeContentSearchCriterionDefinition;
import org.ametys.web.frontoffice.search.metamodel.impl.ContentSearchCriterionDefinition;

/**
 * {@link ContentSearchCriterionDefinition} for a degree content attribute.
 */
public class DegreeUniversityAttributeContentSearchCriterionDefinition extends ContentAttributeContentSearchCriterionDefinition
{
    /** The attribute for degree type */
    public static final String ATTRIBUTE_DEGREE_TYPE = "type";
    
    /** The value for university for degree attribute */
    public static final String ATTRIBUTE_DEGREE_TYPE_UNIVERSITY_VALUE = "UNIVERSITY";

    /** The id of the degree search criterion */
    public static final String DEGREE_SEARCH_CRITERION_ID = ProgramSearchable.PROGRAM_SEARCHEABLE_INDEXING_FIELD_PREFIX + AbstractProgram.DEGREE;
    
    /** The id of the degree university search criterion */
    public static final String DEGREE_UNIVERSITY_SEARCH_CRITERION_ID = DEGREE_SEARCH_CRITERION_ID + "$University";
    
    /** The value for university for degree attribute */
    public static final String SEARCH_CRITERION_ALL_DU_VALUE = "_Ametys_Degree_All_University";
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The degree university helper */
    protected DegreeUniversityHelper _degreeUniversityHelper;
    
    /**
     * Default constructor
     * @param pluginName The plugin name
     * @param searchable The {@link Searchable}
     * @param criterion The linked {@link SearchUICriterion}
     * @param contentType The content type on which this criterion definition applies. Can be empty if it applies to all types of contents.
     * @param validator The validator
     * @param resolver The resolver
     * @param contentTypeEP The extension point for content types
     * @param contentHelper the content helper
     * @param i18nUtils The i18n utils
     * @param degreeUniversityHelper The degree university helper
     */
    public DegreeUniversityAttributeContentSearchCriterionDefinition(
            String pluginName,
            Optional<Searchable> searchable,
            SearchUICriterion criterion,
            Optional<ContentType> contentType,
            Optional<Validator> validator,
            AmetysObjectResolver resolver,
            ContentTypeExtensionPoint contentTypeEP, 
            ContentHelper contentHelper,
            I18nUtils i18nUtils,
            DegreeUniversityHelper degreeUniversityHelper)
    {
        super(DEGREE_UNIVERSITY_SEARCH_CRITERION_ID, pluginName, searchable, criterion, contentType, validator, resolver, contentTypeEP, contentHelper);
        
        _i18nUtils = i18nUtils;
        _degreeUniversityHelper = degreeUniversityHelper;
        setLabel(new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_SEARCH_CRITERION_DEGREE_UNIVERSITY_LABEL"));
    }
    
    @Override
    public Map<Object, I18nizableText> getEntries(String language)
    {
        List<Pair<String, String>> contentsAsPair = _degreeUniversityHelper.getDegrees(false)
            .stream()
            .map(content -> Pair.of(content.getId(), content.getTitle(new Locale(language))))
            .collect(Collectors.toList());
        
        if (!_degreeUniversityHelper.getDegrees(true).isEmpty())
        {
            I18nizableText criterionAllDULabel = new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_SEARCH_CRITERION_DEGREE_UNIVERSITY_FIELD_ALL");
            contentsAsPair.add(Pair.of(SEARCH_CRITERION_ALL_DU_VALUE, _i18nUtils.translate(criterionAllDULabel, language)));
        }
        
        return contentsAsPair.stream()
            .sorted(Comparator.comparing(Pair::getRight)) // sort by title
            .collect(LambdaUtils.Collectors.toLinkedHashMap(Pair::getLeft, pair -> new I18nizableText(pair.getRight())));
    }
    
    @Override
    public Long getOrder(String contentId)
    {
        if (SEARCH_CRITERION_ALL_DU_VALUE.equals(contentId))
        {
            return _degreeUniversityHelper.getDegrees(true)
                .stream()
                .map(c -> c.<Long>getValue("order"))
                .filter(l -> l != null)
                .min(Long::compare)
                .orElse(null);
        }
        else
        {
            return super.getOrder(contentId);
        }
    }
}

