/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.ObservationConstants;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.cache.CacheInvalidationPolicy;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * {@link Observer} for observing OrgUnit validation, unpublishing, deletion or moving
 * (with event {@link ObservationConstants#EVENT_CONTENT_MODIFIED}, which will lead to sometimes invalidating cache whereas it is not necessary)
 *  in order to invalidate cache on front-office.
 */
public class InvalidateCacheOnOrgunitChangedObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** The site manager. */
    protected SiteManager _siteManager;
    
    /** The ODF page handler. */
    protected OdfPageHandler _odfPageHandler;
    
    /** Cache invalidation policy */
    protected CacheInvalidationPolicy _cachePolicy;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _odfPageHandler = (OdfPageHandler) serviceManager.lookup(OdfPageHandler.ROLE);
        _cachePolicy = (CacheInvalidationPolicy) serviceManager.lookup(CacheInvalidationPolicy.ROLE);
    }

    @Override
    public boolean supports(Event event)
    {
        return event.getArguments().get(ObservationConstants.ARGS_CONTENT) instanceof OrgUnit
                && (event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED)
                || event.getId().equals(ObservationConstants.EVENT_CONTENT_DELETING)
                || event.getId().equals(ObservationConstants.EVENT_CONTENT_UNTAG_LIVE)
                || event.getId().equals(ObservationConstants.EVENT_CONTENT_MODIFIED /* for handling moved orgUnit */));
    }

    @Override
    public int getPriority()
    {
        return MAX_PRIORITY + 4000;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        OrgUnit orgUnit = (OrgUnit) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        
        for (Site site : _siteManager.getSites())
        {
            if (_odfPageHandler.hasOdfRootPage(site))
            {
                try
                {
                    _cachePolicy.invalidateCacheOnContentModification(site, orgUnit);
                }
                catch (Exception e)
                {
                    getLogger().error("Unable to invalidate cache for OrgUnit {}", orgUnit, e);
                }
            }
        }
    }
}
