/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.repository.page.Page;

import com.google.common.base.Splitter;

/**
 * {@link AmetysObjectFactory} handling {@link ProgramPage}.
 */
public class CoursePageFactory extends AbstractOdfPageFactory implements AmetysObjectFactory<CoursePage>
{
    /**
     * Create a course page.
     * @param root the odf root page.
     * @param course the course.
     * @param parentProgram the parent program
     * @param path path from the parent {@link ProgramPage}
     * @param parentPage the parent {@link Page} or null if not yet computed.
     * @return The <code>CoursePage</code> created
     */
    public CoursePage createCoursePage(Page root, Course course, Program parentProgram, String path, Page parentPage)
    {
        return new CoursePage(root, getConfiguration(), this, course, parentProgram, path, parentPage);
    }
    
    @Override
    public CoursePage getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // Id is like // E.g: course://path/from/program?rootId=xxx&courseId=xxx&programId=xxxx
        String path = StringUtils.substringBetween(id, "://", "?");
        String queryString = StringUtils.substringAfter(id, "?");
        Map<String, String> ids = Splitter.on("&").withKeyValueSeparator("=").split(queryString);
        
        String rootId = ids.get("rootId");
        String courseId = ids.get("courseId");
        String programId = ids.get("programId");
        
        try
        {
            Page root = _resolver.resolveById(rootId);
            Course course = _resolver.resolveById(courseId);
            Program program = _resolver.resolveById(programId);
            
            // Test program restriction
            if (!_odfPageHandler.isValidRestriction(root, program))
            {
                throw new UnknownAmetysObjectException("There's no course page " + courseId + " for site " + root.getSiteName());
            }
            
            // Check that parent ProgramPage or CoursePage exists
            String levelPath = _odfPageHandler.computeLevelsPath(root, program);
            
            String coursePath = StringUtils.isNotBlank(levelPath) ? levelPath + "/" + path : path;
            root.getChild(coursePath);
            
            try
            {
                CoursePage coursePage = createCoursePage(root, course, program, path, null);
           
                // Test if the virtual page really exists
                coursePage.getPathInSitemap();
                
                return coursePage;
            }
            catch (UnknownAmetysObjectException e)
            {
                throw new UnknownAmetysObjectException("There's no course page " + programId + " for site " + root.getSiteName(), e);
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            throw new UnknownAmetysObjectException("There's no object for id " + id, e);
        }
    }
    
    @Override
    public String getScheme()
    {
        return "course";
    }

    @Override
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        String path = StringUtils.substringBetween(id, "://", "?");
        String queryString = StringUtils.substringAfter(id, "?");
        Map<String, String> ids = Splitter.on("&").withKeyValueSeparator("=").split(queryString);
        
        String rootId = ids.get("rootId");
        String courseId = ids.get("courseId");
        String programId = ids.get("programId");
        
        try
        {
            Page root = _resolver.resolveById(rootId);
            
            Program program = _resolver.resolveById(programId);
            
            // Check that parent ProgramPage exists
            String levelPath = _odfPageHandler.computeLevelsPath(root, program);
            String coursePath = StringUtils.isNotBlank(levelPath) ? levelPath + "/" + path : path;
            root.getChild(coursePath);
            
            return _resolver.hasAmetysObjectForId(courseId);
        }
        catch (UnknownAmetysObjectException e)
        {
            return false;
        }
    }
}
