/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.actions;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.Page;

/**
 * Action providing the list of existing catalogs
 */
public class GetCatalogList extends ServiceableAction
{
    /** The ODF root page handler. */
    protected OdfPageHandler _odfRootHandler;
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _odfRootHandler = (OdfPageHandler) serviceManager.lookup(OdfPageHandler.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String id = request.getParameter("id");
        Page page = _resolver.resolveById(id);
        
        // List of catalogs used by other ODF pages
        Set<String> unavailableCatalogs = 
            _odfRootHandler.getOdfRootPages(page.getSiteName(), page.getSitemapName()).stream()
                .filter(odfRootpage -> !page.equals(odfRootpage))
                .map(_odfRootHandler::getCatalog)
                .filter(StringUtils::isNotEmpty)
                .collect(Collectors.toSet());
        
        // Retrieve the list of available catalogs
        List<Map<String, Object>> catalogList = _odfRootHandler.getCatalogs().entrySet().stream()
            .filter(catalogEntry -> !unavailableCatalogs.contains(catalogEntry.getKey()))
            .map(this::getCatalogData)
            .collect(Collectors.toList());
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, catalogList);
        
        return EMPTY_MAP;
    }
    
    /**
     * Generate a catalog object from its entry
     * @param catalogEntry The catalog entry
     * @return Catalog object containing its data
     */
    protected Map<String, Object> getCatalogData(Entry<String, I18nizableText> catalogEntry)
    {
        Map<String, Object> data = new HashMap<>();
        data.put("code", catalogEntry.getKey());
        data.put("label", catalogEntry.getValue());
        return data;
    }
}
