/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation.solr;

import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.content.indexing.solr.observation.ObserverHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.web.ObservationConstants;
import org.ametys.web.indexing.observation.AbstractSolrObserver;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * {@link Observer} for observing site configuration modifications
 * in order to synchronize Solr index.
 */
public abstract class AbstractSolrOnSiteConfModifiedObserver extends AbstractSolrObserver implements Contextualizable
{
    private OdfPageHandler _odfPageHandler;

    private Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfPageHandler = (OdfPageHandler) manager.lookup(OdfPageHandler.ROLE);
    }

    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_SITE_UPDATED);
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        if (ObserverHelper.isNotSuspendedObservationForIndexation())
        {
            // indicate that we are currently indexing, to allow further optimizations
            Request request = ContextHelper.getRequest(_context);
            request.setAttribute(AbstractSolrODFObserver.REQUEST_ATTRIBUTE_INDEXING, true);

            Site site = (Site) event.getArguments().get(ObservationConstants.ARGS_SITE);
            if (site != null)
            {
                _updateOdfRootIndex(site);
            }
        }
    }

    private void _updateOdfRootIndex(Site site) throws Exception
    {
        String siteName = site.getName();
        for (Sitemap sitemap : site.getSitemaps())
        {
            for (Page odfRootPage : _odfPageHandler.getOdfRootPages(siteName, sitemap.getSitemapName()))
            {
                _updateIndex(odfRootPage);
            }
        }
    }

    /**
     * Update the index from the observed event. 
     * @param odfRootPage the odf root page
     * @throws Exception if an error occurs.
     */
    protected abstract void _updateIndex(Page odfRootPage) throws Exception;
}
