/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.restrictions.rules;

import java.util.Collection;
import java.util.Collections;
import java.util.Objects;

import org.apache.commons.lang3.ArrayUtils;

import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;

/**
 * And restriction rule
 */
public class OdfAndRestrictionRule implements OdfRestrictionRule
{
    private final Collection<OdfRestrictionRule> _rules;
    
    /**
     * And restriction rule constructor
     * @param rules The list of rules on which a "and" operator must be applied
     */
    public OdfAndRestrictionRule(Collection<OdfRestrictionRule> rules)
    {
        _rules = rules;
        _rules.removeAll(Collections.<OdfRestrictionRule>singleton(null));
    }
    
    @Override
    public boolean contains(Program program)
    {
        return _rules.stream().allMatch(r -> r.contains(program));
    }
    
    @Override
    public Expression getExpression()
    {
        Expression[] exprs = _rules.stream()
            .map(OdfRestrictionRule::getExpression)
            .filter(Objects::nonNull)
            .toArray(Expression[]::new);
        
        if (ArrayUtils.isEmpty(exprs))
        {
            return null;
        }
        
        if (exprs.length == 1)
        {
            return exprs[0];
        }
        
        return new AndExpression(exprs);
    }
    
    @Override
    public boolean hasOrgunitRestrictions()
    {
        return _rules.stream().anyMatch(OdfRestrictionRule::hasOrgunitRestrictions);
    }
}
