/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.cart;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesStorage;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;


/**
 * Specific storage for odf cart user preferences
 */
public class ODFCartUserPreferencesStorage extends AbstractLogEnabled implements UserPreferencesStorage, Configurable, Component
{
    /** The Avalon Role */
    public static final String ROLE = ODFCartUserPreferencesStorage.class.getName();
    
    /** The id of the data source used. */
    protected String _dataSourceId;
    
    /** The login column, cannot be null. */
    protected String _loginColumn;
    
    /** The population id column, cannot be null. */
    protected String _populationColumn;
    
    /** The context column, can be null if the database is not context-dependent. */
    protected String _contextColumn;
    
    /** The content id column */
    protected String _contentIdColumn;
    
    /** Mapping from preference id to table name. */
    protected Map<String, String> _prefIdToTable;

    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        // Data source id
        Configuration dataSourceConf = configuration.getChild("datasource", false);
        if (dataSourceConf == null)
        {
            throw new ConfigurationException("The 'datasource' configuration node must be defined.", dataSourceConf);
        }
        
        String dataSourceConfParam = dataSourceConf.getValue();
        String dataSourceConfType = dataSourceConf.getAttribute("type", "config");
        
        if (StringUtils.equals(dataSourceConfType, "config"))
        {
            _dataSourceId = Config.getInstance().getValue(dataSourceConfParam);
        }
        else // expecting type="id"
        {
            _dataSourceId = dataSourceConfParam;
        }
        
        // Default to "contentId".
        _contentIdColumn = configuration.getChild("content-id").getValue("contentId");
        // Default to "login".
        _loginColumn = configuration.getChild("loginColumn").getValue("login").toLowerCase();
        // Default to "population"
        _populationColumn = configuration.getChild("populationColumn").getValue("population").toLowerCase();
        // Default to 'context' (no context column).
        _contextColumn = configuration.getChild("contextColumn").getValue("context");
        
        // Configure the preference-table mappings.
        configureMappings(configuration.getChild("mappings"));
    }
    
    /**
     * Configure the mappings from preference ID to table name.
     * @param configuration the mapping configuration root.
     * @throws ConfigurationException if an error occurs.
     */
    public void configureMappings(Configuration configuration) throws ConfigurationException
    {
        _prefIdToTable = new HashMap<>();
        
        for (Configuration mappingConf : configuration.getChildren("mapping"))
        {
            String prefId = mappingConf.getAttribute("prefId");
            String table = mappingConf.getAttribute("table");
            
            _prefIdToTable.put(prefId, table);
        }
    }
    
    @Override
    public Map<String, String> getUnTypedUserPrefs(UserIdentity user, String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        Map<String, String> prefs = new HashMap<>();
        
        for (String id : _prefIdToTable.keySet())
        {
            prefs.put(id, getUserPreferenceAsString(user, storageContext, contextVars, id));
        }
        
        return prefs;
    }
    
    /**
     * Remove the stored user preferences for a login in a given context.
     * @param prefIds the id of user preferences to remove
     * @param user the user.
     * @param storageContext the preferences storage context.
     * @param contextVars the context variables.
     * @throws UserPreferencesException if an error occurred
     */
    public void removeUserPreferences(Collection<String> prefIds, UserIdentity user, String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        Connection connection = null;
        try
        {
            connection = ConnectionHelper.getConnection(_dataSourceId);
            for (String id : prefIds)
            {
                _removeUserPreferencesFromTable(connection, user, storageContext, _prefIdToTable.get(id));
            }
        }
        catch (SQLException e)
        {
            String message = "Database error trying to remove ODF cart preferences for user '" + user + "' in context '" + storageContext + "'.";
            getLogger().error(message, e);
            throw new UserPreferencesException(message, e);
        }
        finally
        {
            ConnectionHelper.cleanup(connection);
        }
    }
    
    @Override
    public void removeUserPreferences(UserIdentity user, String storageContext, Map<String, String> contextVars) throws UserPreferencesException
    {
        Connection connection = null;
        try
        {
            connection = ConnectionHelper.getConnection(_dataSourceId);
            for (String id : _prefIdToTable.keySet())
            {
                _removeUserPreferencesFromTable(connection, user, storageContext, _prefIdToTable.get(id));
            }
        }
        catch (SQLException e)
        {
            String message = "Database error trying to remove ODF cart preferences for user '" + user + "' in context '" + storageContext + "'.";
            getLogger().error(message, e);
            throw new UserPreferencesException(message, e);
        }
        finally
        {
            ConnectionHelper.cleanup(connection);
        }
    }
    
    private void _removeUserPreferencesFromTable(Connection connection, UserIdentity user, String storageContext, String table) throws SQLException
    {
        PreparedStatement stmt = null;
        
        try
        {
            StringBuilder query = new StringBuilder();
            query.append("DELETE FROM ").append(table).append(" WHERE ").append(_loginColumn).append(" = ? AND ").append(_populationColumn).append(" = ?");
            if (storageContext != null)
            {
                query.append(" AND ").append(_contextColumn).append(" = ?");
            }
            
            stmt = connection.prepareStatement(query.toString());
            stmt.setString(1, user.getLogin());
            stmt.setString(2, user.getPopulationId());
            if (storageContext != null)
            {
                stmt.setString(3, storageContext);
            }
            
            stmt.executeUpdate();
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
        }
        
    }

    /**
     * Remove content from all user preferences
     * @param contentId the content id to remove
     * @throws UserPreferencesException if failed to remove user preferences
     */
    public void removeContentFromUserPreferences(String contentId) throws UserPreferencesException
    {
        Connection connection = null;
        PreparedStatement stmt = null;
        
        try
        {
            connection = ConnectionHelper.getConnection(_dataSourceId);
            
            StringBuilder query = new StringBuilder();
            query.append("DELETE FROM ").append(_prefIdToTable.get(ODFCartManager.CART_USER_PREF_CONTENT_IDS)).append(" WHERE ").append(_contentIdColumn).append(" like ?");
            
            stmt = connection.prepareStatement(query.toString());
            
            stmt.setString(1, contentId + "%");
            stmt.executeUpdate();
            
            ConnectionHelper.cleanup(stmt);
            
            query = new StringBuilder();
            query.append("DELETE FROM ").append(_prefIdToTable.get(ODFCartManager.SUBSCRIPTION_USER_PREF_CONTENT_IDS)).append(" WHERE ").append(_contentIdColumn).append(" like ?");
            
            stmt = connection.prepareStatement(query.toString());
            
            stmt.setString(1, contentId + "%");
            stmt.executeUpdate();
            
        }
        catch (SQLException e)
        {
            String message = "Database error trying to remove ODF content of id '" + contentId + "' from all user's preferences";
            getLogger().error(message, e);
            throw new UserPreferencesException(message, e);
        }
        finally
        {
            ConnectionHelper.cleanup(stmt);
            ConnectionHelper.cleanup(connection);
        }
    }

    @Override
    public void setUserPreferences(UserIdentity user, String storageContext, Map<String, String> contextVars, Map<String, String> preferences) throws UserPreferencesException
    {
        removeUserPreferences(preferences.keySet(), user, storageContext, contextVars);
        
        Connection connection = null;
        try
        {
            connection = ConnectionHelper.getConnection(_dataSourceId);
            _insertPreferences(connection, preferences, user, storageContext);
        }
        catch (SQLException e)
        {
            String message = "Database error trying to set ODF cart preferences of user '" + user + "' in context '" + storageContext + "'.";
            getLogger().error(message, e);
            throw new UserPreferencesException(message, e);
        }
        finally
        {
            ConnectionHelper.cleanup(connection);
        }
    }
    
    private void _insertPreferences(Connection connection, Map<String, String> preferences, UserIdentity user, String storageContext) throws SQLException
    {
        for (String id : preferences.keySet())
        {
            String table = _prefIdToTable.get(id);
            
            String[] contentIdsTab = StringUtils.split(preferences.get(id), ",");
            
            if (contentIdsTab.length > 0)
            {
                PreparedStatement stmt = null;
                try
                {
                    StringBuilder query = new StringBuilder();
                    query.append("INSERT INTO ").append(table).append("(").append(_loginColumn).append(", ").append(_populationColumn);
                    query.append(", ").append(_contextColumn);
                    query.append(", ").append(_contentIdColumn);
    
                    StringBuilder values = new StringBuilder();
                    for (int i = 0; i < contentIdsTab.length; i++)
                    {
                        if (i != 0)
                        {
                            values.append(",");
                        }
                        values.append("(?, ?, ?, ?)");
                    }
                    
                    query.append(") VALUES ").append(values);
                    
                    int i = 1;
                    stmt = connection.prepareStatement(query.toString());
                    
                    for (String idContent : contentIdsTab)
                    {
                        stmt.setString(i++, user.getLogin());
                        stmt.setString(i++, user.getPopulationId());
                        stmt.setString(i++, storageContext);
                        stmt.setString(i++, idContent);
                    }
                    
                    stmt.executeUpdate();
                }
                finally
                {
                    ConnectionHelper.cleanup(stmt);
                }
            }
        }
        
    }

    @Override
    public String getUserPreferenceAsString(UserIdentity user, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        Connection connection = null;
        PreparedStatement statement = null;
        ResultSet rs = null;
        String value = null;
        String table = _prefIdToTable.get(id);
        
        try
        {
            StringBuilder query = new StringBuilder();
            query.append("SELECT ").append(_contentIdColumn).append(" FROM ").append(table).append(" WHERE ").append(_loginColumn).append(" = ? AND ").append(_populationColumn).append(" = ?");
            query.append(" AND ").append(_contextColumn).append(" = ?");
            
            connection = ConnectionHelper.getConnection(_dataSourceId);
            
            statement = connection.prepareStatement(query.toString());
            statement.setString(1, user.getLogin());
            statement.setString(2, user.getPopulationId());
            statement.setString(3, storageContext);
            
            rs = statement.executeQuery();
            
            List<String> results = new ArrayList<>();
            while (rs.next())
            {
                results.add(rs.getString(1));
            }
            
            value = StringUtils.join(results, ",");
        }
        catch (SQLException e)
        {
            String message = "Database error trying to get ODF cart preferences of user '" + user + "' in context '" + storageContext + "'.";
            getLogger().error(message, e);
            throw new UserPreferencesException(message, e);
        }
        finally
        {
            ConnectionHelper.cleanup(rs);
            ConnectionHelper.cleanup(statement);
            ConnectionHelper.cleanup(connection);
        }
        
        return value;
    }
    
    @Override
    public Long getUserPreferenceAsLong(UserIdentity user, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        return null;
    }

    @Override
    public ZonedDateTime getUserPreferenceAsDate(UserIdentity user, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        return null;
    }

    @Override
    public Boolean getUserPreferenceAsBoolean(UserIdentity user, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        return null;
    }

    @Override
    public Double getUserPreferenceAsDouble(UserIdentity user, String storageContext, Map<String, String> contextVars, String id) throws UserPreferencesException
    {
        return null;
    }

}
