/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.Set;

import org.apache.commons.lang.StringUtils;

import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.UnknownZoneException;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * This class is used for legacy purpose only
 * Old program's page's name is built with program's title and program's name such as 'master-droits-program-fruai1452h45g81'.
 * For program page requested with a name such as 'master-droits-program-fruai1452h45g81', a page of redirection will be served
 * to the new program's page url built with program's title and program's code such as 'master-droits-H45G81'.
 */
public class RedirectPage implements Page
{
    private Page _page;

    /**
     * Constructor for redirect page to a {@link Page} for legacy purpose
     * @param page the program or course page
     */
    public RedirectPage(Page page)
    {
        _page = page;
    }
    
    /**
     * Returns the wrapped {@link Page}.
     * @return the wrapped page
     */
    public Page getRedirectPage()
    {
        return _page;
    }
    
    public PageType getType() throws AmetysRepositoryException
    {
        return PageType.LINK;
    }
    
    public String getURL() throws AmetysRepositoryException
    {
        return _page.getId();
    }

    public LinkType getURLType() throws AmetysRepositoryException
    {
        return LinkType.PAGE;
    }

    public Sitemap getSitemap() throws AmetysRepositoryException
    {
        return _page.getSitemap();
    }

    public String getSitemapName() throws AmetysRepositoryException
    {
        return _page.getSitemapName();
    }

    public String getPathInSitemap() throws AmetysRepositoryException
    {
        return getParent().getPathInSitemap() + "/" + getName();
    }

    public AmetysObjectIterable< ? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        return _addRedirects(_page.getChildrenPages());
    }

    public AmetysObjectIterable< ? extends Page> getChildrenPages(boolean includeInvisiblePages) throws AmetysRepositoryException
    {
        return _addRedirects(_page.getChildrenPages(includeInvisiblePages));
    }

    public Page getChildPageAt(int index) throws UnknownAmetysObjectException, AmetysRepositoryException
    {
        return _addRedirect(_page.getChildPageAt(index));
    }

    public <A extends AmetysObject> AmetysObjectIterable<A> getChildren() throws AmetysRepositoryException
    {
        return _addRedirects(_page.getChildren());
    }

    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        return _page.hasChild(name);
    }

    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        return _addRedirect(_page.getChild(path));
    }

    public String getName() throws AmetysRepositoryException
    {
        return _page.getName();
    }

    public String getPath() throws AmetysRepositoryException
    {
        return getParentPath() + "/" + getName();
    }

    public String getId() throws AmetysRepositoryException
    {
        return StringUtils.replace(_page.getId(), "program://", "unused://");
    }
    
    public String getTitle() throws AmetysRepositoryException
    {
        return _page.getTitle();
    }

    public String getLongTitle() throws AmetysRepositoryException
    {
        return _page.getLongTitle();
    }

    @SuppressWarnings("unchecked")
    public Page getParent() throws AmetysRepositoryException
    {
        return _page.getParent();
    }

    public String getParentPath() throws AmetysRepositoryException
    {
        return _page.getParentPath();
    }

    public String getSiteName() throws AmetysRepositoryException
    {
        return _page.getSiteName();
    }

    public Site getSite() throws AmetysRepositoryException
    {
        return _page.getSite();
    }

    public ModelLessDataHolder getDataHolder()
    {
        return _page.getDataHolder();
    }

    public Set<String> getTags() throws AmetysRepositoryException
    {
        return _page.getTags();
    }

    public int getDepth() throws AmetysRepositoryException
    {
        return _page.getDepth();
    }

    public String getTemplate() throws AmetysRepositoryException
    {
        return _page.getTemplate();
    }

    public AmetysObjectIterable< ? extends Zone> getZones() throws AmetysRepositoryException
    {
        return _page.getZones();
    }

    public boolean hasZone(String name) throws AmetysRepositoryException
    {
        return _page.hasZone(name);
    }

    public Zone getZone(String name) throws UnknownZoneException, AmetysRepositoryException
    {
        return _page.getZone(name);
    }

    public ResourceCollection getRootAttachments() throws AmetysRepositoryException
    {
        throw new UnsupportedOperationException("getRootAttachments not supported on legacy redirect odf pages");
    }

    public Set<String> getReferers() throws AmetysRepositoryException
    {
        throw new UnsupportedOperationException("getReferers not supported on legacy redirect odf pages");
    }

    public boolean isVisible() throws AmetysRepositoryException
    {
        return false;
    }

    public boolean isIndexable() throws AmetysRepositoryException
    {
        return false;
    }
    
    public ModelAwareDataHolder getTemplateParametersHolder() throws AmetysRepositoryException
    {
        return null;
    }
    
    @SuppressWarnings("unchecked")
    private <A extends AmetysObject> A _addRedirect(AmetysObject ametysObject)
    {
        return (A) (
                ametysObject instanceof Page page
                ? new RedirectPage(
                    page instanceof RedirectPage redirectPage
                    ? redirectPage.getRedirectPage()
                    : page
                )
                : ametysObject
            );
    }
    
    private <A extends AmetysObject> AmetysObjectIterable<A> _addRedirects(AmetysObjectIterable<? extends AmetysObject> iterable)
    {
        return new CollectionIterable<>(
            iterable.stream()
                .map(this::<A>_addRedirect)
                .toList()
        );
    }
}
