/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfweb.repository;

import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.repository.page.Page;

import com.google.common.base.Splitter;

/**
 * {@link AmetysObjectFactory} handling {@link ProgramPage}.
 */
public class ProgramPageFactory extends AbstractOdfPageFactory implements AmetysObjectFactory<ProgramPage>
{
    /**
     * Create a program page.
     * @param root the ODF root page.
     * @param program the program or subprogram.
     * @param path The path from the virtual second level page. Can be null if abstract program is a {@link Program}
     * @param parent the parent program in case of a subprogram, null otherwise
     * @param parentPage the parent {@link Page} or null if not yet computed.
     * @return The <code>ProgramPage</code> created
     */
    public ProgramPage createProgramPage(Page root, AbstractProgram program, String path, Program parent, Page parentPage) 
    {
        return new ProgramPage(root, getConfiguration(), this, program, path, parent, parentPage);
    }
    
    @Override
    public ProgramPage getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // Id is like program://_root?rootId=xxxx&programId=xxxx for Program
        // Id is like program://path/to/subprogram?rootId=xxxx&programId=xxxx&parentId=xxxx for SubProgram
        
        String path = StringUtils.substringBetween(id, "://", "?");
        String queryString = StringUtils.substringAfter(id, "?");
        Map<String, String> ids = Splitter.on("&").withKeyValueSeparator("=").split(queryString);
        
        String rootId = ids.get("rootId");
        String programId = ids.get("programId");
        String parentId = ids.get("parentId");
        
        if ("_root".equals(path))
        {
            // Case of a program
            path = null;
        }
        
        Page root = _resolver.resolveById(rootId);
        AbstractProgram program = _resolver.resolveById(programId); // program or subprogram
        Program parent = StringUtils.isNotEmpty(parentId) ? _resolver.resolveById(parentId) : null;
        Program parentProgramOrSelf = _getParentProgramOrSelf(parent, program);
        
        // The identifier is invalid
        if (parentProgramOrSelf == null)
        {
            throw new UnknownAmetysObjectException("The object '" + program.getId() + "' is not a Program and its given parent is null.");
        }
        
        // Test program restriction
        if (!_odfPageHandler.isValidRestriction(root, parentProgramOrSelf))
        {
            throw new UnknownAmetysObjectException("There's no program child page " + programId + " for site " + root.getSiteName());
        }
        
        try 
        {
            ProgramPage programPage = createProgramPage(root, program, path, parent, null);
       
            // Test if the virtual page really exists
            programPage.getPathInSitemap();
            
            return programPage;
        }
        catch (UnknownAmetysObjectException e)
        {
            throw new UnknownAmetysObjectException("There's no program child page " + programId + " for site " + root.getSiteName(), e);
        }
    }
    
    private Program _getParentProgramOrSelf(Program parent, AbstractProgram self) throws UnknownAmetysObjectException
    {
        if (parent != null)
        {
            return parent;
        }
        
        // In old identifiers, a subprogram target doesn't need the parentId, so if we don't have a parent, it can be an old target with a subprogram
        if (self instanceof Program)
        {
            return (Program) self;
        }
        
        return null;
    }
    
    @Override
    public String getScheme()
    {
        return "program";
    }

    @Override
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        int i = id.indexOf('?');
            
        String queryString = id.substring(i + 1);
        Map<String, String> ids = Splitter.on("&").withKeyValueSeparator("=").split(queryString);
        
        String rootId = ids.get("rootId");
        String programId = ids.get("programId");
        String parentId = ids.get("parentId");
        
        if (!_resolver.hasAmetysObjectForId(rootId) || !_resolver.hasAmetysObjectForId(programId))
        {
            return false;
        }
        
        Page root = _resolver.resolveById(rootId);
        AbstractProgram program = _resolver.resolveById(programId); // program or subprogram
        Program parent = StringUtils.isNotEmpty(parentId) ? _resolver.resolveById(parentId) : null;
        Program parentProgramOrSelf = _getParentProgramOrSelf(parent, program);

        // The identifier is invalid
        if (parentProgramOrSelf == null)
        {
            return false;
        }
        
        if (!_odfPageHandler.isValidRestriction(root, parentProgramOrSelf))
        {
            return false;
        }
        
        if (StringUtils.isNotBlank(_odfPageHandler.getLevel2Metadata(root)))
        {
            return _odfPageHandler.getProgramLevel1Value(root, parentProgramOrSelf) != null && _odfPageHandler.getProgramLevel2Value(root, parentProgramOrSelf) != null;
        }
        else if (StringUtils.isNotBlank(_odfPageHandler.getLevel1Metadata(root)))
        {
            return _odfPageHandler.getProgramLevel1Value(root, parentProgramOrSelf) != null;
        }
        
        return true;
    }
}
