/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.inputdata;

import java.util.Map;

import org.ametys.cms.repository.Content;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.translation.TranslationHelper;
import org.ametys.plugins.odfweb.repository.CoursePage;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.odfweb.repository.ProgramPage;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.inputdata.InputData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

/**
 * {@link InputData} which generates the translations of program and course pages.
 */
public class TranslationsInputData extends AbstractLogEnabled implements InputData, Serviceable, Contextualizable
{
    /** Avalon context. */
    protected Context _context;
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;
    
    /** The ODF page path resolver. */ 
    protected OdfPageResolver _odfPageResolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _ametysResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfPageResolver = (OdfPageResolver) manager.lookup(OdfPageResolver.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public boolean isCacheable(Site site, Page page)
    {
        return true;
    }
    
    @Override
    public void toSAX(ContentHandler handler) throws SAXException, ProcessingException
    {
        Request request = ContextHelper.getRequest(_context);
        Page page = (Page) request.getAttribute(Page.class.getName());
        
        handler.startDocument();
        XMLUtils.startElement(handler, "translations");
        
        if (page instanceof ProgramPage)
        {
            AbstractProgram program = ((ProgramPage) page).getProgram();
            
            Map<String, String> translations = TranslationHelper.getTranslations(program);
            _saxTranslations (handler, translations);
        }
        else if (page instanceof CoursePage)
        {
            Course course = ((CoursePage) page).getCourse();
            
            Map<String, String> translations = TranslationHelper.getTranslations(course);
            _saxTranslations (handler, translations);
        }
        
        XMLUtils.endElement(handler, "translations");
        handler.endDocument();
    }
    
    private void _saxTranslations (ContentHandler contentHandler, Map<String, String> translations) throws SAXException
    {
        for (String lang : translations.keySet())
        {
            String contentId = translations.get(lang);
            try
            {
                Content translatedProgram = _ametysResolver.resolveById(contentId);
                
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("language", lang);
                attrs.addCDATAAttribute("id", translatedProgram.getId());
                
                XMLUtils.createElement(contentHandler, "translation", attrs);
            }
            catch (UnknownAmetysObjectException e)
            {
                // Ignore, the translation just doesn't exist in the live workspace.
            }
        }
    }
}
