/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import java.util.Collection;
import java.util.Set;
import java.util.function.Function;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.search.query.Query;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.web.frontoffice.search.metamodel.AdditionalParameterValueMap;
import org.ametys.web.frontoffice.search.metamodel.Returnable;
import org.ametys.web.frontoffice.search.metamodel.ReturnableExtensionPoint;
import org.ametys.web.frontoffice.search.metamodel.ReturnableSaxer;
import org.ametys.web.frontoffice.search.metamodel.impl.AbstractContentBasedReturnable;
import org.ametys.web.frontoffice.search.metamodel.impl.PageReturnable;

/**
 * {@link Returnable} for {@link Program}s
 */
public class ProgramReturnable extends AbstractContentBasedReturnable
{
    /** Avalon Role */
    public static final String ROLE = ProgramReturnable.class.getName();
    
    /** The additional parameter for indicating how subprograms are displayed */
    public static final String PARAMETER_DISPLAY_SUBPROGRAMS = "displaySubprogram";
    
    /** The prefix for the ids of sorts */
    protected static final String __SORT_DEFINITION_PREFIX_ID = "ProgramReturnable$";

    /** The context */
    Context _avalonContext;
    
    private OdfPageResolver _odfPageResolver;
    private ContentTypeExtensionPoint _cTypeEP;
    private ReturnableExtensionPoint _returnableEP;
    private PageReturnable _pageReturnable;
    
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfPageResolver = (OdfPageResolver) manager.lookup(OdfPageResolver.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _returnableEP = (ReturnableExtensionPoint) manager.lookup(ReturnableExtensionPoint.ROLE);
    }
    
    @Override
    protected Function<Query, Query> siteQueryJoiner()
    {
        // Programs contents are "no-site" contents
        return null;
    }
    
    /**
     * Get the ODF page resolver
     * @return the ODF page resolver
     */
    protected OdfPageResolver _getOdfPageResolver()
    {
        return _odfPageResolver;
    }
    
    /**
     * Get the {@link ContentTypeExtensionPoint}
     * @return the {@link ContentTypeExtensionPoint}
     */
    protected ContentTypeExtensionPoint _getContentTypeEP()
    {
        return _cTypeEP;
    }
    
    /**
     * Get the {@link Returnable} for page
     * @return the {@link PageReturnable}
     */
    protected PageReturnable _gePageReturnable()
    {
        if (_pageReturnable == null)
        {
            _pageReturnable = (PageReturnable) _returnableEP.getExtension(PageReturnable.ROLE);
        }
        return _pageReturnable;
    }
    
    @Override
    protected String associatedContentSearchableRole()
    {
        return ProgramSearchable.ROLE;
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        super.contextualize(context);
        _avalonContext = _context;
    }
    
    @Override
    public String getId()
    {
        return ROLE;
    }
    
    @Override
    protected Set<String> getContentTypeIds(AdditionalParameterValueMap additionalParameterValues)
    {
        return Set.of(ProgramFactory.PROGRAM_CONTENT_TYPE);
    }
    
    @Override
    public ReturnableSaxer getSaxer(Collection<Returnable> allReturnables, AdditionalParameterValueMap additionalParameterValues)
    {
        DisplaySubprogramMode displaySubprogramMode = getDisplaySubprogramMode(additionalParameterValues);
        return new ProgramSaxer(_gePageReturnable(), this, displaySubprogramMode);
    }
    
    static DisplaySubprogramMode getDisplaySubprogramMode(AdditionalParameterValueMap additionalParameterValues)
    {
        String displaySubprogramParamValue = additionalParameterValues.getValue(PARAMETER_DISPLAY_SUBPROGRAMS);
        return DisplaySubprogramMode.valueOf(displaySubprogramParamValue.toUpperCase());
    }
    
    @Override
    protected String getSortDefinitionPrefix()
    {
        return __SORT_DEFINITION_PREFIX_ID;
    }
    
    static enum DisplaySubprogramMode
    {
        /** Display no subprogram */
        NONE,
        /** Display all subprograms */
        ALL,
        /** Display all subprograms with highlighting those which match the search criteria */
        ALL_WITH_HIGHLIGHT,
        /** Display matching subprograms only */
        MATCHING_SEARCH_ONLY
    }
}
