/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation.solr;

import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.content.indexing.solr.observation.ObserverHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.odfweb.restrictions.OdfProgramRestrictionManager;
import org.ametys.web.indexing.observation.AbstractSolrObserver;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.site.SiteConfigurationExtensionPoint;

/**
 * {@link Observer} for observing orgunit modifications
 * in order to synchronize Solr index.
 */
public abstract class AbstractSolrOrgUnitObserver extends AbstractSolrObserver implements Contextualizable
{
    /** The site manager. */
    protected SiteManager _siteManager;
    /** The site configuration. */
    protected SiteConfigurationExtensionPoint _siteConf;
    /** The ODF page handler. */
    protected OdfPageHandler _odfPageHandler;
    /** Odf program restriction manager */
    protected OdfProgramRestrictionManager _odfProgramRestrictionManager;
    
    private Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _odfPageHandler = (OdfPageHandler) serviceManager.lookup(OdfPageHandler.ROLE);
        _siteConf = (SiteConfigurationExtensionPoint) serviceManager.lookup(SiteConfigurationExtensionPoint.ROLE);
        _odfProgramRestrictionManager = (OdfProgramRestrictionManager) serviceManager.lookup(OdfProgramRestrictionManager.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getArguments().get(ObservationConstants.ARGS_CONTENT) instanceof OrgUnit && ArrayUtils.contains(_supportedEventIds(), event.getId());
    }
    
    /**
     * Gets the supported event ids for this Observer to look for the {@link OrgUnit}s
     * @return the supported event ids for this Observer
     */
    protected abstract String[] _supportedEventIds();

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        if (ObserverHelper.isNotSuspendedObservationForIndexation())
        {
            // indicate that we are currently indexing, to allow further optimizations
            Request request = ContextHelper.getRequest(_context);
            request.setAttribute(AbstractSolrODFObserver.REQUEST_ATTRIBUTE_INDEXING, true);

            for (Site site : _siteManager.getSites())
            {
                String siteName = site.getName();
                if (_odfPageHandler.hasOdfRootPage(site))
                {
                    for (Sitemap sitemap : site.getSitemaps())
                    {
                        for (Page odfRootPage : _odfPageHandler.getOdfRootPages(siteName, sitemap.getSitemapName()))
                        {
                            if (_odfProgramRestrictionManager.hasOrgunitRestrictions(odfRootPage)
                                || ProgramItem.ORG_UNITS_REFERENCES.equals(_odfPageHandler.getLevel1Metadata(odfRootPage))
                                || ProgramItem.ORG_UNITS_REFERENCES.equals(_odfPageHandler.getLevel2Metadata(odfRootPage)))
                            {
                                _updateIndex(odfRootPage);
                            }
                        }
                    }
                }
            }
        }
    }

    /**
     * Update the index from the observed event. 
     * @param odfRootPage the odf root page
     * @throws Exception if an error occurs.
     */
    protected abstract void _updateIndex(Page odfRootPage) throws Exception;
}
