/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Button controller related to the ODF root page
 */
Ext.define('Ametys.plugins.odf.web.root.ODFRootButtonController', {
    extend: 'Ametys.web.controller.WebButtonController',
    
    constructor: function(config)
    {
        this.callParent(arguments);
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
    },
    
    /**
     * Listener handler for modified messages
     * @param {Ametys.message.Message} message the message
     */
     _onModified: function(message)
     { 
         if (this.updateTargetsInCurrentSelectionTargets(message))
         {
             this.refresh();
         }
     },
     
     updateState: function()
     {
        this._getStatus(this.getMatchingTargets());
     },
     
    /**
     * @private
     * Get the status
     * @param targets The page targets
     */
    _getStatus: function (targets)
    {
        this.disable();
        
        if (targets && targets.length == 1)
        {
            var pageId = targets[0].getParameters().id;
            
            this.serverCall('getStatus',
                [pageId], 
                Ext.bind(this._getStatusCb, this),
                {
                    errorMessage: true,
                    refreshing: true
                }
            );
        }
    },
    
    /**
     * Callback to refresh button data after the status has been retrieved.
     * @param {Object} params parameters from the the server's response
     */
    _getStatusCb: function(params)
    {
        this._updateTooltipDescription(params);
        
        this.toggle(params['is-odf-root'] === true);
        this.setDisabled(params['invalid-page'] === true);
    },
    
    /**
     * @private
     * Update the tooltip description according state of the current selection
     * @param {Object} params The JSON result received
     */
    _updateTooltipDescription: function(params)
    {
        var description = '';
        
        if (params['is-odf-root'] === true)
        {
           description = this._addToDescription(description, params['odf-root-page-description']);
           description = this._addToDescription(description, params['catalog-description'], ' ');
           description = this._addToDescription(description, params['remove-odf-page-description']);
        }
        else if (params['add-odf-page'] === true)
        {
           description = this._addToDescription(description, params['add-odf-page-description']);
        }
        else if (params['invalid-page'] === true)
        {
           description = this._addToDescription(description, params['no-jcr-page-description']);
        }
        
        this.setAdditionalDescription(description);
    },
    
    /**
     * @private
     * Add text to description
     * @param {String} descAccumulator The string containing the accumulated description
     * @param {String} newPart A new part of description to add to the accumulator
     * @param {String} [separator] Optional separator that will be inserted before the addition of the new part. Will default to '<br/><br/>'.
     * @return {String} The concatenated description
     */
    _addToDescription: function(descAccumulator, newPart, separator)
    {
        if (!newPart)
        {
            return;
        }
        
        if (descAccumulator)
        {
            descAccumulator += (separator || '<br/><br/>');
        }
        
        descAccumulator += newPart;
        return descAccumulator;
    }
});
