/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search.criterion;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.data.ContentValue;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.plugins.odfweb.service.search.ProgramSearchable;
import org.ametys.plugins.odfweb.service.search.helper.DegreeUniversityHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Enumerator;
import org.ametys.web.frontoffice.search.metamodel.RestrictedEnumerator;
import org.ametys.web.frontoffice.search.metamodel.impl.ContentReferencingSearchServiceCriterionDefinition;
import org.ametys.web.frontoffice.search.metamodel.impl.RestrictedWrappedEnumerator;

/**
 * {@link ContentReferencingSearchServiceCriterionDefinition} for a degree content attribute.
 */
public class DegreeUniversityCriterionDefinition extends ContentReferencingSearchServiceCriterionDefinition
{
    /** The attribute for degree type */
    public static final String ATTRIBUTE_DEGREE_TYPE = "type";
    
    /** The value for university for degree attribute */
    public static final String ATTRIBUTE_DEGREE_TYPE_UNIVERSITY_VALUE = "UNIVERSITY";

    /** The name of the degree criterion definition */
    public static final String DEGREE_CRITERION_DEFINITION_NAME = ProgramSearchable.PROGRAM_SEARCHEABLE_CRITERION_DEFINITION_PREFIX + AbstractProgram.DEGREE;
    
    /** The name of the degree university criterion definition */
    public static final String DEGREE_UNIVERSITY_CRITERION_DEFINITION_NAME = DEGREE_CRITERION_DEFINITION_NAME + "$University";
    
    /** The value for university for degree attribute */
    public static final String CRITERION_ALL_DU_VALUE = "_Ametys_Degree_All_University";
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The degree university helper */
    protected DegreeUniversityHelper _degreeUniversityHelper;
    
    /**
     * Constructor used to create a FO criterion definition referencing the degree university
     * @param reference the item referenced by this criterion
     * @param referencePath the path of the criterion's reference
     * @param baseContentType the content type defining the reference
     */
    public DegreeUniversityCriterionDefinition(ElementDefinition reference, String referencePath, ContentType baseContentType)
    {
        super(reference, referencePath, baseContentType);
    }
    
    @Override
    public String getName()
    {
        return DEGREE_UNIVERSITY_CRITERION_DEFINITION_NAME;
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_SEARCH_CRITERION_DEGREE_UNIVERSITY_LABEL");
    }
    
    @Override
    public boolean isEnumerated()
    {
        return true;
    }
    
    @Override
    public RestrictedEnumerator<ContentValue> getRestrictedEnumerator(Map<String, Object> contextualParameters)
    {
        Enumerator<ContentValue> enumerator = new DegreeUniversityEnumerator(this, contextualParameters);
        return new RestrictedWrappedEnumerator<>(enumerator, getName());
    }
    
    @Override
    public Long getOrder(ContentValue contentValue)
    {
        if (contentValue == null)
        {
            return _getDegreeUniversityHelper().getDegrees(true)
                .stream()
                .map(c -> c.<Long>getValue("order"))
                .filter(l -> l != null)
                .min(Long::compare)
                .orElse(null);
        }
        else
        {
            return super.getOrder(contentValue);
        }
    }
    

    
    /**
     * Retrieves the {@link I18nUtils} 
     * @return the {@link I18nUtils}
     */
    protected I18nUtils _getI18nUtils()
    {
        if (_i18nUtils == null)
        {
            try
            {
                _i18nUtils = (I18nUtils) __serviceManager.lookup(I18nUtils.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException("Unable to lookup after the I18n utils component", e);
            }
        }
        
        return _i18nUtils;
    }
    
    /**
     * Retrieves the {@link DegreeUniversityHelper} 
     * @return the {@link DegreeUniversityHelper}
     */
    protected DegreeUniversityHelper _getDegreeUniversityHelper()
    {
        if (_degreeUniversityHelper == null)
        {
            try
            {
                _degreeUniversityHelper = (DegreeUniversityHelper) __serviceManager.lookup(DegreeUniversityHelper.ROLE);
            }
            catch (ServiceException e)
            {
                throw new RuntimeException("Unable to lookup after the degree university helper", e);
            }
        }
        
        return _degreeUniversityHelper;
    }
    
    static class DegreeUniversityEnumerator implements Enumerator<ContentValue>
    {
        DegreeUniversityCriterionDefinition _criterionDefinition;
        Map<String, Object> _contextualParameters;
        
        DegreeUniversityEnumerator(DegreeUniversityCriterionDefinition criterionDefinition, Map<String, Object> contextualParameters)
        {
            _criterionDefinition = criterionDefinition;
            _contextualParameters = contextualParameters;
        }
        
        public Map<ContentValue, I18nizableText> getEntries() throws Exception
        {
            String language = (String) _contextualParameters.get("lang");
            
            @SuppressWarnings("synthetic-access")
            List<Pair<ContentValue, String>> contentsAsPair = _criterionDefinition._getDegreeUniversityHelper().getDegrees(false)
                    .stream()
                    .map(content -> Pair.of(new ContentValue(_criterionDefinition._getAmetysObjectResolver(), content.getId()), content.getTitle(LocaleUtils.toLocale(language))))
                    .collect(Collectors.toList());
                
            if (!_criterionDefinition._getDegreeUniversityHelper().getDegrees(true).isEmpty())
            {
                I18nizableText criterionAllDULabel = new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_SEARCH_CRITERION_DEGREE_UNIVERSITY_FIELD_ALL");
                contentsAsPair.add(Pair.of(null, _criterionDefinition._getI18nUtils().translate(criterionAllDULabel, language)));
            }
            
            return contentsAsPair.stream()
                .sorted(Comparator.comparing(Pair::getRight)) // sort by title
                .collect(LambdaUtils.Collectors.toLinkedHashMap(Pair::getLeft, pair -> new I18nizableText(pair.getRight())));
        }
    }
}

