/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.catalog;

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.DirectoryStream.Filter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.acting.AbstractAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.web.source.SkinSource;

/**
 * Get available InDesign template files from the current skin
 */
public class GetInDesignTemplatesAction extends AbstractAction
{
    private static InDesignPathFilter __PATH_FILTER = new InDesignPathFilter(); 
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> templates = new ArrayList<>();
        
        String skinName = request.getParameter("skinName");
        if (StringUtils.isNotEmpty(skinName))
        {
            SkinSource indesignDirSource = (SkinSource) resolver.resolveURI("skin:" + skinName + "://indesign");
            
            if (indesignDirSource.exists() && indesignDirSource.isCollection())
            {
                // Use getFile() to get only the first existing path
                try (DirectoryStream<Path> models = Files.newDirectoryStream(indesignDirSource.getFile(), __PATH_FILTER))
                {
                    for (Path model : models)
                    {
                        String fileName = model.getFileName().toString();
                        
                        Map<String, Object> info = new HashMap<>();
                        info.put("filepath", fileName);
                        info.put("filename", fileName.substring(0, fileName.lastIndexOf(".")));
                        
                        templates.add(info);
                    }
                }
            }
        }
        
        result.put("templates", templates);
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    static class InDesignPathFilter implements Filter<Path>
    {
        public boolean accept(Path entry) throws IOException
        {
            return Files.isRegularFile(entry) && entry.getFileName().toString().toLowerCase().endsWith(".indt");
        }
    }
}
