/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.cachepolicy;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfweb.OdfWebObservationConstants;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.ObservationConstants;
import org.ametys.web.cache.pageelement.PageElementCachePolicy;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * Cache policy for the program list service.
 */
public class OdfProgramListServiceCachePolicy extends AbstractLogEnabled implements Serviceable, PageElementCachePolicy
{
    
    private static final Set<String> _TYPES = new HashSet<>();
    static
    {
        _TYPES.add("SERVICE:org.ametys.odf.service.ProgramList");
    }
    
    /** The ODF root page handler. */
    protected OdfPageHandler _odfRootPageHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfRootPageHandler = (OdfPageHandler) manager.lookup(OdfPageHandler.ROLE);
    }
    
    @Override
    public Set<String> getPageElementTypes()
    {
        return _TYPES;
    }
    
    @Override
    public final PolicyResult shouldClearCache(String workspace, Site site, String pageElementType, Event event)
    {
        String id = event.getId();
        
        if (_getRemovingCacheEventIds(workspace).contains(id))
        {
            Map<String, Object> args = event.getArguments();
            
            if (args.containsKey(org.ametys.cms.ObservationConstants.ARGS_CONTENT))
            {
                Object object = args.get(org.ametys.cms.ObservationConstants.ARGS_CONTENT);
                if (object instanceof Program || object instanceof OrgUnit)
                {
                    if (_odfRootPageHandler.hasOdfRootPage(site))
                    {
                        return PolicyResult.REMOVE;
                    }
                }
            }
            
            if (args.get(ObservationConstants.ARGS_PAGE) instanceof Page page1 && _odfRootPageHandler.isODFRootPage(page1) 
                || args.get(ObservationConstants.ARGS_SITEMAP_ELEMENT) instanceof Page page2 && _odfRootPageHandler.isODFRootPage(page2))
            {
                return PolicyResult.REMOVE;
            }
        }
        
        return PolicyResult.KEEP;
    }
    
    @Override
    public final PolicyResult shouldClearCache(String workspace, Site site, String pageElementType, String elementId, Event event)
    {
        // Never called because the first-level method never returns NEED_INFORMATION.
        throw new UnsupportedOperationException("Should never be called.");
    }
    
    /**
     * Returns all event ids for which the cache should be removed.
     * @param workspace the current JCR workspace.
     * @return all event ids for which the cache should be removed.
     */
    protected List<String> _getRemovingCacheEventIds(String workspace)
    {
        if ("default".equals(workspace))
        {
            return Arrays.asList(org.ametys.cms.ObservationConstants.EVENT_CONTENT_ADDED,
                                 org.ametys.cms.ObservationConstants.EVENT_CONTENT_MODIFIED,
                                 org.ametys.cms.ObservationConstants.EVENT_CONTENT_DELETING,
                                 OdfWebObservationConstants.ODF_ROOT_UPDATED,
                                 OdfWebObservationConstants.ODF_ROOT_DELETED);
        }
        else if ("live".equals(workspace))
        {
            return Arrays.asList(org.ametys.cms.ObservationConstants.EVENT_CONTENT_ADDED,
                    org.ametys.cms.ObservationConstants.EVENT_CONTENT_VALIDATED,
                    org.ametys.cms.ObservationConstants.EVENT_CONTENT_DELETING,
                    OdfWebObservationConstants.ODF_ROOT_UPDATED,
                    OdfWebObservationConstants.ODF_ROOT_DELETED);
        }
        
        return Collections.emptyList();
    }
}
