/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.cart;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.userpref.FOUserPreferencesConstants;

import jakarta.mail.MessagingException;

/**
 * Component to handle ODF cart items
 *
 */
public class ODFCartManager extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** The avalon role */
    public static final String ROLE = ODFCartManager.class.getName();
    
    /** The id of user preference for cart's elements */
    public static final String CART_USER_PREF_CONTENT_IDS = "cartOdfContentIds";
    
    /** The id of user preference for subscription */
    public static final String SUBSCRIPTION_USER_PREF_CONTENT_IDS = "subscriptionOdfContentIds";
    
    private UserPreferencesManager _userPrefManager;
    private AmetysObjectResolver _resolver;
    private SourceResolver _srcResolver;
    private OdfPageResolver _odfPageResolver;
    private ContentTypeExtensionPoint _cTypeEP;
    private ContentTypesHelper _cTypesHelper;
    private ODFCartUserPreferencesStorage _odfUserPrefStorage;
    private I18nUtils _i18nUtils;
    private UserManager _userManager;
    private SiteManager _siteManager;
    private RenderingContextHandler _renderingContextHandler;
    private CurrentUserProvider _currentUserProvider;

    private Context _context;

    private ContentHelper _contentHelper;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _userPrefManager = (UserPreferencesManager) serviceManager.lookup(UserPreferencesManager.ROLE + ".FO");
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _srcResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _odfPageResolver = (OdfPageResolver) serviceManager.lookup(OdfPageResolver.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _cTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
        _odfUserPrefStorage = (ODFCartUserPreferencesStorage) serviceManager.lookup(ODFCartUserPreferencesStorage.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _renderingContextHandler = (RenderingContextHandler) serviceManager.lookup(RenderingContextHandler.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the id of ODF's cart items for a given user
     * @param user the user
     * @param siteName the current site name
     * @return the list of contents' id
     * @throws UserPreferencesException if failed to get cart items
     */
    public List<String> getCartItemIds(UserIdentity user, String siteName) throws UserPreferencesException
    {
        return getCartItemIds(user, siteName, CART_USER_PREF_CONTENT_IDS);
    }
    
    /**
     * Get the id of ODF's cart items for which a given user is a subscriber
     * @param user the user
     * @param siteName the current site name
     * @return the list of contents' id
     * @throws UserPreferencesException if failed to get cart items
     */
    public List<String> getItemIdsWithSubscription(UserIdentity user, String siteName) throws UserPreferencesException
    {
        return getCartItemIds(user, siteName, SUBSCRIPTION_USER_PREF_CONTENT_IDS);
    }
    
    /**
     * Get the id of items for a given user
     * @param user the user
     * @param siteName the current site name
     * @param userPrefsId The id of user preferences
     * @return the list of contents' id
     * @throws UserPreferencesException if failed to get cart items
     */
    protected List<String> getCartItemIds(UserIdentity user, String siteName, String userPrefsId) throws UserPreferencesException
    {
        Map<String, String> contextVars = new HashMap<>();
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, siteName);
        
        String contentIdsAsStr = _userPrefManager.getUserPreferenceAsString(user, "/sites/" + siteName, contextVars, userPrefsId);
        if (StringUtils.isNotBlank(contentIdsAsStr))
        {
            return Arrays.asList(StringUtils.split(contentIdsAsStr, ","));
        }
        
        return Collections.emptyList();
    }
    
    /**
     * Get the ODF's cart items for a given user
     * @param owner the user
     * @param siteName the current site name
     * @return the list of contents
     * @throws UserPreferencesException if failed to get cart items
     */
    public List<ODFCartItem> getCartItems(UserIdentity owner, String siteName) throws UserPreferencesException
    {
        List<ODFCartItem> items = new ArrayList<>();
        
        List<String> itemIds = getCartItemIds(owner, siteName);
        for (String itemId : itemIds)
        {
            ODFCartItem item = getCartItem(itemId);
            if (item != null)
            {
                items.add(item);
            }
            else
            {
                getLogger().warn("The item with id '{}' stored in cart of user {} does not match an existing content anymore. It will be ignored", itemId, owner);
            }
        }
        
        return items;
    }
    
    /**
     * Get the ODF's cart items for which the given user is a subscriber
     * @param owner the user
     * @param siteName the current site name
     * @return the list of contents
     * @throws UserPreferencesException if failed to get subscriptions
     */
    public List<ODFCartItem> getItemsWithSubscription(UserIdentity owner, String siteName) throws UserPreferencesException
    {
        List<ODFCartItem> items = new ArrayList<>();
        
        List<String> itemIds = getItemIdsWithSubscription(owner, siteName);
        for (String itemId : itemIds)
        {
            ODFCartItem item = getCartItem(itemId);
            if (item != null)
            {
                items.add(item);
            }
            else
            {
                getLogger().warn("The item with id '{}' stored in subscription of user {} does not match an existing content anymore. It will be ignored", itemId, owner);
            }
        }
        
        return items;
    }
    
    /**
     * Get a cart item from its id
     * @param itemId the item's id
     * @return the cart item or null if no content was found
     */
    public ODFCartItem getCartItem(String itemId)
    {
        int i = itemId.indexOf(';');
        
        String contentId = itemId;
        String parentId = null;
        
        if (i != -1)
        {
            contentId = itemId.substring(0, i);
            parentId = itemId.substring(i + 1);
        }
        
        try
        {
            return new ODFCartItem(_resolver.resolveById(contentId), parentId != null ? _resolver.resolveById(parentId) : null);
        }
        catch (UnknownAmetysObjectException e)
        {
            return null;
        }
    }
    
    /**
     * Set the cart's items
     * @param owner The cart owner
     * @param itemIds the id of items to set in the cart
     * @param siteName the site name
     * @throws UserPreferencesException if failed to save cart
     */
    public void setCartItems(UserIdentity owner, List<String> itemIds, String siteName) throws UserPreferencesException
    {
        saveItemsInUserPreference(owner, itemIds, siteName, CART_USER_PREF_CONTENT_IDS);
    }
    
    
    /**
     * Subscribe to a list of items. All subscriptions are replaced by the given items.
     * @param owner The cart owner
     * @param itemIds the id of items to subscribe to.
     * @param siteName the site name
     * @throws UserPreferencesException if failed to save subscriptions
     */
    public void setSubscription(UserIdentity owner, List<String> itemIds, String siteName) throws UserPreferencesException
    {
        saveItemsInUserPreference(owner, itemIds, siteName, SUBSCRIPTION_USER_PREF_CONTENT_IDS);
    }
    
    /**
     * Save the given items into the given user preference
     * @param owner the user
     * @param itemIds the id of items
     * @param siteName the site name
     * @param userPrefsId the id of user preference
     * @throws UserPreferencesException if failed to save user preference
     */
    public void saveItemsInUserPreference(UserIdentity owner, List<String> itemIds, String siteName, String userPrefsId) throws UserPreferencesException
    {
        Map<String, String> contextVars = new HashMap<>();
        contextVars.put(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME, siteName);
        
        Map<String, String> preferences = new HashMap<>();
        preferences.put(userPrefsId, StringUtils.join(itemIds, ","));
        
        _odfUserPrefStorage.setUserPreferences(owner, "/sites/" + siteName, contextVars, preferences);
    }
    
    /**
     * Add a content to the cart
     * @param owner the cart owner
     * @param itemId the id of content to add into the cart
     * @param siteName the site name
     * @return true if the content was successfully added
     * @throws UserPreferencesException if failed to save cart
     */
    public boolean addCartItem(UserIdentity owner, String itemId, String siteName) throws UserPreferencesException
    {
        List<ODFCartItem> items = getCartItems(owner, siteName);
        
        ODFCartItem item = getCartItem(itemId);
        if (item != null)
        {
            items.add(item);
        }
        else
        {
            getLogger().warn("Unknown item with id {}. It cannot be added to user cart", itemId);
            return false;
        }
        
        List<String> itemIds = items.stream()
                .map(c -> c.getId())
                .collect(Collectors.toList());
        
        setCartItems(owner, itemIds, siteName);
        
        return true;
    }
    
    /**
     * determines if the user subscribes to this item
     * @param owner the user
     * @param itemId the id of content
     * @param siteName the site name
     * @return true if the user subscribes to this item, false otherwise
     * @throws UserPreferencesException if failed to check subscription
     */
    public boolean isSubscriber(UserIdentity owner, String itemId, String siteName) throws UserPreferencesException
    {
        List<String> subscriptions = getItemIdsWithSubscription(owner, siteName);
        return subscriptions.contains(itemId);
    }
    
    /**
     * Subscribe to a content
     * @param owner the cart owner
     * @param itemId the id of content
     * @param siteName the site name
     * @return if the content was successfuly added to subscriptions
     * @throws UserPreferencesException if failed to subscribe to content
     */
    public boolean subscribe(UserIdentity owner, String itemId, String siteName) throws UserPreferencesException
    {
        List<ODFCartItem> items = getItemsWithSubscription(owner, siteName);
        
        ODFCartItem item = getCartItem(itemId);
        if (item != null)
        {
            items.add(item);
        }
        else
        {
            getLogger().warn("Unknown item with id {}. It cannot be added to user subscriptions", itemId);
            return false;
        }
        
        List<String> itemIds = items.stream()
                .map(c -> c.getId())
                .collect(Collectors.toList());
        
        setSubscription(owner, itemIds, siteName);
        
        return true;
    }
    
    /**
     * Unsubscribe to a content
     * @param owner the cart owner
     * @param itemId the id of content
     * @param siteName the site name
     * @return if the content was successfuly added to subscriptions
     * @throws UserPreferencesException if failed to subscribe to content
     */
    public boolean unsubscribe(UserIdentity owner, String itemId, String siteName) throws UserPreferencesException
    {
        List<ODFCartItem> items = getItemsWithSubscription(owner, siteName);
        
        ODFCartItem item = getCartItem(itemId);
        if (item != null)
        {
            items.remove(item);
        }
        else
        {
            getLogger().warn("Unknown item with id {}. It cannot be remove from user subscriptions", itemId);
            return false;
        }
        
        List<String> itemIds = items.stream()
                .map(c -> c.getId())
                .collect(Collectors.toList());
        
        setSubscription(owner, itemIds, siteName);
        
        return true;
    }
    
    /**
     * Share the cart's items by mail
     * @param owner The cart owner
     * @param itemIds the id of contents to set in the cart
     * @param recipients the mails to share with
     * @param siteName the site name
     * @param language the default language
     * @param message the message to add to selection
     * @return the results
     */
    public Map<String, Object> shareCartItems(UserIdentity owner, List<String> itemIds, List<String> recipients, String siteName, String language, String message)
    {
        Map<String, Object> result = new HashMap<>();
        
        User user = _userManager.getUser(owner);
        String lang = StringUtils.defaultIfBlank(user.getLanguage(), language);
        String sender = user.getEmail();
        
        if (StringUtils.isEmpty(sender))
        {
            getLogger().error("Cart's owner has no email, his ODF cart selection can not be shared");
            result.put("success", false);
            result.put("error", "no-owner-mail");
            return result;
        }
        
        List<ODFCartItem> items = itemIds.stream()
                .map(i -> getCartItem(i))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        
        Site site = _siteManager.getSite(siteName);
        Map<String, I18nizableTextParameter> i18nparam = new HashMap<>();
        i18nparam.put("siteTitle", new I18nizableText(site.getTitle())); // {siteTitle}
        
        I18nizableText i18nTextSubject = new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_CART_SHARE_MAIL_SUBJECT", i18nparam);
        String subject = _i18nUtils.translate(i18nTextSubject, lang);
        
        String htmlBody = null;
        String textBody = null;
        try
        {
            
            htmlBody = getMailBody(items, message, owner, siteName, lang, false);
            textBody = getMailBody(items, message, owner, siteName, lang, true);
        }
        catch (IOException e)
        {
            getLogger().error("Fail to get mail body to share ODF cart selection", e);
            result.put("success", false);
            return result;
        }
        
        List<String> mailsInError = new ArrayList<>();
        
        for (String recipient : recipients)
        {
            try
            {
                String prettyHTMLBody;
                try
                {
                    prettyHTMLBody = StandardMailBodyHelper.newHTMLBody()
                            .withLanguage(lang)
                            .withTitle(subject)
                            .withMessage(htmlBody.replaceAll("\n", "")) // remove breaklines
                            .withLink(site.getUrl(), new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_CART_SHARE_MAIL_HTML_BODY_SITE_LINK"))
                            .build();
                }
                catch (IOException e)
                {
                    getLogger().warn("Failed to build wrapped HTML body for ODF cart. Fallback to no wrapped mail", e);
                    prettyHTMLBody = htmlBody;
                }
                
                
                SendMailHelper.newMail()
                              .withSubject(subject)
                              .withHTMLBody(prettyHTMLBody)
                              .withTextBody(textBody)
                              .withRecipient(recipient)
                              .withSender(sender)
                              .sendMail();
            }
            catch (MessagingException | IOException e)
            {
                getLogger().error("Failed to send ODF cart selection to '" + recipient + "'", e);
                mailsInError.add(recipient);
            }
        }
        
        if (mailsInError.size() > 0)
        {
            result.put("success", false);
            result.put("mailsInError", mailsInError);
        }
        else
        {
            result.put("success", true);
        }
        
        return result;
    }
    
    /**
     * Get the mail subject for sharing cart
     * @param siteName The site name
     * @param language the language
     * @return the mail subject
     */
    protected String getMailSubject(String siteName, String language)
    {
        Site site = _siteManager.getSite(siteName);
        Map<String, I18nizableTextParameter> i18nparam = new HashMap<>();
        i18nparam.put("siteTitle", new I18nizableText(site.getTitle())); // {siteTitle}
        
        I18nizableText i18nTextSubject = new I18nizableText("plugin.odf-web", "PLUGINS_ODFWEB_CART_SHARE_MAIL_SUBJECT", i18nparam);
        return _i18nUtils.translate(i18nTextSubject, language);
    }
    
    /**
     * Get the mail body to sharing cart
     * @param items The cart's items
     * @param message The message
     * @param owner The cart's owner
     * @param siteName The site name
     * @param language the language
     * @param text true to get the body to text body (html otherwise)
     * @return the cart items to HTML format
     * @throws IOException if failed to mail body
     */
    protected String getMailBody(List<ODFCartItem> items, String message, UserIdentity owner, String siteName, String language, boolean text) throws IOException
    {
        Request request = ContextHelper.getRequest(_context);
        
        String currentWorkspace = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
        
        Source source = null;
        try
        {
            // Force live workspace and FRONT context to resolve page
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, WebConstants.LIVE_WORKSPACE);
            _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
            
            Map<String, Object> parameters = new HashMap<>();
            
            parameters.put("items", items);
            parameters.put("message", message);
            parameters.put("owner", owner);
            parameters.put("siteName", siteName);
            parameters.put("lang", language);
            parameters.put("format", text ? "text" : "html");
            
            source = _srcResolver.resolveURI("cocoon://_plugins/odf-web/cart/mail/body", null, parameters);
            
            try (InputStream is = source.getInputStream())
            {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                SourceUtil.copy(is, bos);
                
                return bos.toString("UTF-8");
            }
        }
        finally
        {
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWorkspace);
            _renderingContextHandler.setRenderingContext(currentContext);
            
            if (source != null)
            {
                _srcResolver.release(source);
            }
        }
    }
    
    /**
     * SAX the cart's items
     * @param contentHandler The content handler to sax into
     * @param owner the cart owner
     * @param siteName the site name
     * @throws SAXException if an error occurred while saxing
     * @throws IOException if an I/O exception occurred
     * @throws UserPreferencesException if failed to get cart items
     */
    public void saxCartItems(ContentHandler contentHandler, UserIdentity owner, String siteName) throws SAXException, IOException, UserPreferencesException
    {
        List<ODFCartItem> items = getCartItems(owner, siteName);
        
        XMLUtils.startElement(contentHandler, "items");
        for (ODFCartItem item : items)
        {
            saxCartItem(contentHandler, item, siteName);
            
        }
        XMLUtils.endElement(contentHandler, "items");
        
    }
    
    /**
     * SAX a cart's item
     * @param contentHandler The content handler to sax into
     * @param item the cart's item
     * @param siteName the site name
     * @throws SAXException if an error occurred while saxing
     * @throws IOException if an I/O exception occurred
     */
    public void saxCartItem(ContentHandler contentHandler, ODFCartItem item, String siteName) throws SAXException, IOException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", item.getId());
        XMLUtils.startElement(contentHandler, "item", attrs);
        
        Content content = item.getContent();
        saxTypes(contentHandler, content.getTypes());
        saxContent(contentHandler, content, "cart");
        saxPage(contentHandler, item, siteName);
        
        Program parentProgram = item.getParentProgram();
        if (parentProgram != null)
        {
            attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", parentProgram.getId());
            attrs.addCDATAAttribute("title", parentProgram.getTitle());
            Page parentPage = _odfPageResolver.getProgramPage(parentProgram, siteName);
            if (parentPage != null)
            {
                attrs.addCDATAAttribute("pageId", parentPage.getId());
            }
            XMLUtils.createElement(contentHandler, "parent", attrs);
            
        }
        XMLUtils.endElement(contentHandler, "item");
    }
    
    /**
     * Get the JSON representation of a cart item
     * @param item The cart's item
     * @param siteName The site name
     * @param viewName The name of content view to use
     * @return The cart items properties
     * @throws IOException if failed to read content view
     */
    public Map<String, Object> cartItem2Json(ODFCartItem item, String siteName, String viewName) throws IOException
    {
        Map<String, Object> result = new HashMap<>();
        
        Content content = item.getContent();
        
        result.put("id", item.getId());
        result.put("contentId", content.getId());
        result.put("title", content.getTitle());
        result.put("name", content.getName());
        
        Program parentProgram = item.getParentProgram();
        if (parentProgram != null)
        {
            result.put("parentProgramId", parentProgram.getId());
            result.put("parentProgramTitle", parentProgram.getTitle());
        }
        
        Page page = getPage(item, siteName);
        if (page != null)
        {
            result.put("pageId", page.getId());
            result.put("pageTitle", page.getTitle());
            result.put("pagePath", page.getPathInSitemap());
        }
        
        String cTypeId = content.getTypes()[0];
        ContentType cType = _cTypeEP.getExtension(cTypeId);
        
        result.put("contentTypeId", cTypeId);
        result.put("contentTypeLabel", cType.getLabel());
        
        if (viewName != null && _cTypesHelper.getView(viewName, content.getTypes(), content.getMixinTypes()) != null)
        {
            String uri = _contentHelper.getContentHtmlViewUrl(content, viewName, parentProgram != null ? Map.of("parentProgramId", parentProgram.getId()) : Map.of());
            SitemapSource src = null;
            
            try
            {
                src = (SitemapSource) _srcResolver.resolveURI(uri);
                try (InputStream is = src.getInputStream())
                {
                    String view = IOUtils.toString(is, StandardCharsets.UTF_8);
                    result.put("view", view);
                }
            }
            finally
            {
                _srcResolver.release(src);
            }
        }
        
        if (content instanceof AbstractProgram)
        {
            try
            {
                boolean subscriber = _currentUserProvider.getUser() != null && isSubscriber(_currentUserProvider.getUser(), item.getId(), siteName);
                result.put("subscriber", subscriber);
            }
            catch (UserPreferencesException e)
            {
                getLogger().error("Fail to check if current user subscribes to content {}. It supposes he is not.", content.getId(), e);
                result.put("subscriber", false);
            }
            
            additionalItemInfo(item, (AbstractProgram) content, result);
        }
        else if (content instanceof Course)
        {
            additionalItemInfo(item, (Course) content, result);
        }
        
        return result;
    }
    
    /**
     * Get the additional information for {@link AbstractProgram} cart item
     * @param item the odf cart item
     * @param abstractProgram the abstract program
     * @param infos the information to be completed
     */
    protected void additionalItemInfo(ODFCartItem item, AbstractProgram abstractProgram, Map<String, Object> infos)
    {
        // Nothing
    }
    
    /**
     * Get the additional information for {@link Course} cart item
     * @param item the odf cart item
     * @param course the abstract program
     * @param infos the information to be completed
     */
    protected void additionalItemInfo(ODFCartItem item, Course course, Map<String, Object> infos)
    {
        double ects = course.getEcts();
        if (ects > 0D)
        {
            infos.put("ects", ects);
        }
        
        double numberOfHours = course.getNumberOfHours();
        if (numberOfHours > 0D)
        {
            infos.put("nbHours", numberOfHours);
        }
        
        List<CoursePart> courseParts = course.getCourseParts();
        if (!courseParts.isEmpty())
        {
            List<Map<String, Object>> courseparts = new ArrayList<>();
            for (CoursePart coursePart : course.getCourseParts())
            {
                Map<String, Object> coursepart = new HashMap<>();
                coursepart.put("nature", coursePart.getNature());
                coursepart.put("nbHours", coursePart.getNumberOfHours());
            }
            
            infos.put("courseparts", courseparts);
        }
    }
    
    /**
     * Sax the content types
     * @param handler The content handler to sax into
     * @param types The content types
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxTypes(ContentHandler handler, String[] types) throws SAXException
    {
        XMLUtils.startElement(handler, "types");
        
        for (String id : types)
        {
            ContentType cType = _cTypeEP.getExtension(id);
            if (cType != null)
            {
                AttributesImpl attrs = new AttributesImpl();
                attrs.addCDATAAttribute("id", cType.getId());
                
                XMLUtils.startElement(handler, "type", attrs);
                cType.getLabel().toSAX(handler);
                XMLUtils.endElement(handler, "type");
            }
        }
        XMLUtils.endElement(handler, "types");
    }
    
    /**
     * SAX the content view
     * @param handler The content handler to sax into
     * @param content The content
     * @param viewName The view name
     * @throws SAXException if an error occurred while saxing
     * @throws IOException if an I/O exception occurred
     */
    protected void saxContent (ContentHandler handler, Content content, String viewName) throws SAXException, IOException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("name", content.getName());
        attrs.addCDATAAttribute("title", content.getTitle(null));
        attrs.addCDATAAttribute("lastModifiedAt", DateUtils.zonedDateTimeToString(content.getLastModified()));
        
        XMLUtils.startElement(handler, "content", attrs);
        
        if (_cTypesHelper.getView("cart", content.getTypes(), content.getMixinTypes()) != null)
        {
            String uri = _contentHelper.getContentHtmlViewUrl(content, viewName);
            SitemapSource src = null;
            
            try
            {
                src = (SitemapSource) _srcResolver.resolveURI(uri);
                src.toSAX(new IgnoreRootHandler(handler));
            }
            finally
            {
                _srcResolver.release(src);
            }
        }
        
        XMLUtils.endElement(handler, "content");
    }
    
    /**
     * Sax the content's page
     * @param handler The content handler to sax into
     * @param item The cart's item
     * @param siteName The current site name
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxPage(ContentHandler handler, ODFCartItem item, String siteName) throws SAXException
    {
        Page page = getPage(item, siteName);
        if (page != null)
        {
            String pageId = page.getId();
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", pageId);
            attrs.addCDATAAttribute("path", ResolveURIComponent.resolve("page", pageId));
            XMLUtils.createElement(handler, "page", attrs, page.getTitle());
        }
    }
    
    /**
     * Get the page associated to this cart's item
     * @param item The item
     * @param siteName The site name
     * @return the page or <code>null</code> if not found
     */
    protected Page getPage(ODFCartItem item, String siteName)
    {
        Content content  = item.getContent();
        if (content instanceof Course)
        {
            return _odfPageResolver.getCoursePage((Course) content, (AbstractProgram) item.getParentProgram(), siteName);
        }
        else if (content instanceof Program)
        {
            return _odfPageResolver.getProgramPage((Program) content, siteName);
        }
        else if (content instanceof SubProgram)
        {
            return _odfPageResolver.getSubProgramPage((SubProgram) content, item.getParentProgram(), siteName);
        }
        
        getLogger().info("No page found of content {} in ODF cart", content.getId());
        return null;
    }
    
    class ODFCartItem
    {
        private Content _content;
        private Program _parentProgram;
        
        public ODFCartItem(Content content)
        {
            this(content, null);
        }
        
        public ODFCartItem(Content content, Program parentProgram)
        {
            _content = content;
            _parentProgram = parentProgram;
        }
        
        String getId()
        {
            return _content.getId() + (_parentProgram != null ? ";" + _parentProgram.getId() : "");
        }
        
        Content getContent()
        {
            return _content;
        }
        
        Program getParentProgram()
        {
            return _parentProgram;
        }
        
        @Override
        public int hashCode()
        {
            return Objects.hash(getId());
        }
        
        @Override
        public boolean equals(Object other)
        {
            return other != null && getClass() == other.getClass() && getId().equals(((ODFCartItem) other).getId());
        }
    }
}
