/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.inputdata;

import java.io.IOException;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfweb.repository.CoursePage;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.inputdata.InputData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;


/**
 * {@link InputData} for SAXing the list of programs having the current course among their courseLists 
 */
public class CourseInputData extends AbstractLogEnabled implements InputData, Contextualizable, Initializable, Serviceable
{
    /** Avalon context. */
    protected Context _context;
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _ametysResolver;
    
    /** CMS Sites manager */ 
    protected SiteManager _siteManager;
    
    /** The root handler */
    protected OdfPageHandler _odfPageHandler;
    
    /** The ODF reference table helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** Configuration boolean to active or not the inputdata */
    private boolean _isActive;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _ametysResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfPageHandler = (OdfPageHandler) manager.lookup(OdfPageHandler.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void initialize() throws Exception
    {
        _isActive = Config.getInstance().getValue("odf.course.showReferences");
    }
    
    @Override
    public boolean isCacheable(Site site, Page page)
    {
        return true;
    }
    
    @Override
    public void toSAX(ContentHandler handler) throws SAXException, ProcessingException
    {
        if (_isActive)
        {
            handler.startDocument();
            XMLUtils.startElement(handler, "programsSearchedByCourse");
            
            Request request = ContextHelper.getRequest(_context);
            
            String siteName = (String) request.getAttribute("site");
            String sitemapLanguage = (String) request.getAttribute("sitemapLanguage");

            Page page = (Page) request.getAttribute(Page.class.getName());
            if (page instanceof CoursePage)
            {
                Course course = ((CoursePage) page).getCourse();
                
                Set<Program> referencingPrograms = _odfHelper.getParentPrograms(course);
                if (!referencingPrograms.isEmpty())
                {
                    Page rootPage = _odfPageHandler.getOdfRootPage(siteName, sitemapLanguage);
                    AttributesImpl atts = new AttributesImpl();
                    if (rootPage != null)
                    {
                        atts.addCDATAAttribute("root-page-path", rootPage.getPathInSitemap());
                    }
                    XMLUtils.startElement(handler, "programs", atts);
                    
                    for (Program program : referencingPrograms)
                    {
                        try
                        {
                            saxProgram(program, handler);
                        }
                        catch (IOException e)
                        {
                            throw new ProcessingException("An error occurs SAXing a program : " + program.getId(), e);
                        }
                    }
                    XMLUtils.endElement(handler, "programs");
                }
            }
            
            XMLUtils.endElement(handler, "programsSearchedByCourse");
            handler.endDocument();
        }
    }

    /**
     * SAX a program.
     * @param program The program to SAX.
     * @param contentHandler the content handler
     * @throws SAXException If an error occurs while SAXing
     * @throws IOException If an error occurs while retrieving content.
     */
    public void saxProgram(Program program, ContentHandler contentHandler) throws SAXException, IOException
    {
        AttributesImpl atts = new AttributesImpl();
        
        List<String> orgUnitIds = program.getOrgUnits();
        String degreeId = program.getDegree();
        String[] dgesipIds = program.getDGESIPCode();
        String[] siseids = program.getSiseCode();
        
        atts.addCDATAAttribute("name", program.getName());
        atts.addCDATAAttribute("title", program.getTitle());
        
        XMLUtils.startElement(contentHandler, "program", atts);

        for (String domain : program.getDomain())
        {
            atts.clear();
            atts.addCDATAAttribute("id", domain);
            atts.addCDATAAttribute("code", _odfRefTableHelper.getItemCode(domain));
            XMLUtils.createElement(contentHandler, "domain", atts);
        }
        
        for (String orgUnitId : orgUnitIds)
        {
            atts.clear();
            atts.addCDATAAttribute("id", orgUnitId);
            XMLUtils.createElement(contentHandler, "orgUnit", atts);
        }
        
        if (StringUtils.isNotEmpty(degreeId))
        {
            atts.clear();
            atts.addCDATAAttribute("id", degreeId);
            atts.addCDATAAttribute("code", _odfRefTableHelper.getItemCode(degreeId));
            XMLUtils.startElement(contentHandler, "degree", atts);
            XMLUtils.endElement(contentHandler, "degree");
        }
        
        if (dgesipIds != null)
        {
            for (String dgesipId : dgesipIds)
            {
                atts.clear();
                atts.addCDATAAttribute("id", dgesipId);
                atts.addCDATAAttribute("code", _odfRefTableHelper.getItemCode(dgesipId));
                XMLUtils.startElement(contentHandler, "dgesip", atts);
                XMLUtils.endElement(contentHandler, "dgesip");
            }
        }
        
        if (siseids != null)
        {
            for (String siseId : siseids)
            {
                atts.clear();
                atts.addCDATAAttribute("code", siseId);
                atts.addCDATAAttribute("code", _odfRefTableHelper.getItemCode(siseId));
                XMLUtils.startElement(contentHandler, "sise", atts);
                XMLUtils.endElement(contentHandler, "sise");
            }
        }
        
        XMLUtils.endElement(contentHandler, "program");
    }    
}
