/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation;

import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.web.WebConstants;
import org.ametys.web.cache.CacheInvalidationPolicy;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * {@link Observer} for observing content validation or tagging in order to
 * invalidate cache on front-office.
 */
public class InvalidateCacheOnContentValidationOrUnpublishingObserver extends AbstractODFObserver
{
    /** Cache invalidation policy */
    protected CacheInvalidationPolicy _cachePolicy;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _cachePolicy = (CacheInvalidationPolicy) serviceManager.lookup(CacheInvalidationPolicy.class.getName());
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED)
            || event.getId().equals(ObservationConstants.EVENT_CONTENT_UNTAG_LIVE);
    }
    
    @Override
    public int getPriority()
    {
        // Will be processed after Solr update
        return MAX_PRIORITY + 4000;
    }
    
    @Override
    protected String _workspaceToUse()
    {
        return WebConstants.LIVE_WORKSPACE;
    }
    
    @Override
    protected void _internalObserve(Event event, Map<String, Object> transientVars, Page odfRootPage, Set<Program> rootPrograms, SubProgram subProgram, Course course) throws Exception
    {
        Site site = odfRootPage.getSite();
        
        for (Program program : rootPrograms)
        {
            _invalidate(site, program);
        }
        
        if (subProgram != null)
        {
            _invalidate(site, subProgram);
        }
        
        if (course != null)
        {
            _invalidate(site, course);
        }
        
    }
    
    /**
     * Invalidate content cache
     * @param site the site
     * @param content the content
     */
    protected void _invalidate(Site site, Content content)
    {
        try
        {
            _cachePolicy.invalidateCacheOnContentModification(site, content);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to invalidate cache for progam {}", content, e);
        }
    }
}
