/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.program.Program;
import org.ametys.odf.tree.OdfClassificationHandler.LevelValue;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

/**
 * Page representing a first-level page.
 */
public class FirstLevelPage extends AbstractOdfPage<FirstLevelPageFactory>
{
    private String _name;
    private String _title;
    
    /**
     * Constructor.
     * @param factory the corresponding factory
     * @param root the odf root page.
     * @param name the page's name.
     * @param title the page's title.
     * @param configuration the configuration
     */
    public FirstLevelPage(Page root, VirtualPageConfiguration configuration, FirstLevelPageFactory factory, String name, String title)
    {
        super(root, configuration, factory.getScheme(), factory);
        
        _name = name;
        _title = title;
    }
    
    @Override
    public int getDepth() throws AmetysRepositoryException
    {
        return _root.getDepth() + 1;
    }

    @Override
    public String getTitle() throws AmetysRepositoryException
    {
        return _title;
    }
    
    @Override
    public String getLongTitle() throws AmetysRepositoryException
    {
        return _title;
    }

    @Override
    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        if (!_hasSecondLevelPages())
        {
            List<ProgramPage> pages = _getChildrenProgramPageFromCache()
                    .orElseGet(() -> _factory.getODFPageHandler().getProgramsWithRestrictions(_root, _name, null, null, null))
                    .stream()
                    .map(this::_createProgramPage)
                    .collect(Collectors.toList());
                
            return new CollectionIterable<>(pages);
                
        }
        else
        {
            Map<String, Collection<Program>> secondLevelCache = Optional.ofNullable(_factory.getODFPageCache().getProgramCache(_root, true).get(_name)).orElseGet(HashMap::new);
            Map<String, LevelValue> secondLevelValues = _factory.getODFPageHandler().getLevel2Values(_root);
            
            List<Page> children = secondLevelCache.keySet().stream()
                    .map(secondLevelCode -> _findSecondLevelValueEntry(secondLevelValues, secondLevelCode).orElse(null))
                    .filter(Objects::nonNull)
                    .map(entry -> _createSecondLevelPage(entry))
                    .collect(Collectors.toList());
            
            return new CollectionIterable<>(children); 
        }
    }
    
    private boolean _hasSecondLevelPages()
    {
        String level2Metadata = _factory.getODFPageHandler().getLevel2Metadata(_root);
        return StringUtils.isNotBlank(level2Metadata);
    }
    
    private Optional<AmetysObjectIterable<Program>> _getChildrenProgramPageFromCache()
    {
        return _factory.getODFPageCache().getPrograms(_root, _name, null, false);
    }
    
    private Optional<Entry<String, LevelValue>> _findSecondLevelValueEntry(String levelCode)
    {
        Map<String, LevelValue> secondLevelValues = _factory.getODFPageHandler().getLevel2Values(_root);
        return _findSecondLevelValueEntry(secondLevelValues, levelCode);
    }
    
    private Optional<Entry<String, LevelValue>> _findSecondLevelValueEntry(Map<String, LevelValue> secondLevelValues, String levelCode)
    {
        return secondLevelValues.entrySet().stream()
             // entry = (code, title)
            .filter(entry -> entry.getKey().equals(levelCode))
            .findFirst();
    }
    
    private SecondLevelPage _createSecondLevelPage(Map.Entry<String, LevelValue> secondLevelValueEntry)
    {
        String childName = secondLevelValueEntry.getKey();
        String title = secondLevelValueEntry.getValue().getValue();
        
        return _factory.getSecondLevelPageFactory().createSecondLevelPage(_root, _name, childName, title, this);
    }
    
    private ProgramPage _createProgramPage(Program program)
    {
        return _factory.getProgramPageFactory().createProgramPage(_root, program, null, null, this);
    }
    
    @Override
    public String getPathInSitemap() throws AmetysRepositoryException
    {
        return _root.getPathInSitemap() + "/" + getName();
    }

    @SuppressWarnings("unchecked")
    @Override
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path.isEmpty())
        {
            throw new AmetysRepositoryException("path must be non empty");
        }
        
        if (!_hasSecondLevelPages())
        {
            return (A) _factory.getODFPageCache().getChildProgramPage(_root, this, _name, null, path);
        }
        else
        {
            int i = path.indexOf('/');
            
            String name = i == -1 ? path : path.substring(0, path.indexOf('/'));
            
            int index = name.lastIndexOf("-");
            String level2Code = index != -1 ? _factory.getODFPageHandler().decodeLevelValue(name.substring(index + 1)) : _factory.getODFPageHandler().decodeLevelValue(name);
            String queuePath = i == -1 ? null : path.substring(i + 1);

            return (A) _findSecondLevelValueEntry(level2Code)
                .map(this::_createSecondLevelPage)
                .map(cp -> _factory.getODFPageHandler().addRedirectIfNeeded(cp, name))
                .map(cp -> _factory.getODFPageHandler().exploreQueuePath(cp, queuePath))
                .orElseThrow(() -> new UnknownAmetysObjectException("Unknown child page '" + path + "' for page " + _name));
        }
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public AmetysObjectIterable<? extends Page> getChildren() throws AmetysRepositoryException
    {
        return getChildrenPages();
    }

    @Override
    public boolean hasChild(String childName) throws AmetysRepositoryException
    {
        // FIXME should use page cache, because can return true when page does
        // not exist. Maybe done like this for performance reasons?
        if (!_hasSecondLevelPages())
        {
            return _factory.getODFPageCache().getProgramFromPageName(_root, _name, null, childName).isPresent();
        }
        else
        {
            Map<String, LevelValue> secondLevelValues = _factory.getODFPageHandler().getLevel2Values(_root);
            return secondLevelValues.containsKey(childName);
        }
    }
    
    @Override
    public String getId() throws AmetysRepositoryException
    {
        // E.g: odfLevel1://XA?rootId=...
        return "odfLevel1://" + _factory.getODFPageHandler().encodeLevelValue(_name) + "?rootId=" + _root.getId();
    }

    @Override
    public String getName() throws AmetysRepositoryException
    {
        // E.g: licence-lmd-XA
        return NameHelper.filterName(_title) + "-" + _factory.getODFPageHandler().encodeLevelValue(_name);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Page getParent() throws AmetysRepositoryException
    {
        return _root;
    }

    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        return _root.getPath();
    }

    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }
}
