/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation.solr;

import java.util.Map;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;

/**
 * <p>
 * {@link Observer} for observing odf content validation in order to synchronize
 * lucene index associated.
 * </p>
 */
public class SolrContentValidatedPart2Observer extends AbstractSolrODFObserver implements AsyncObserver
{
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED);
    }

    @Override
    protected String _workspaceToUse()
    {
        return WebConstants.LIVE_WORKSPACE;
    }
    
    @Override
    protected Content _getContentArgument(Event event) throws Exception
    {
        // Async observer cannot directly use an AmetysObject as argument because its session might not be alive anymore.
        Content content = super._getContentArgument(event);
        if (content == null)
        {
            return null;
        }
        
        // content is resolved in the current workspace (ie. live in this case)
        return _resolver.resolveById(content.getId());
    }
    
    @Override
    protected void _updateIndex(Event event, Map<String, Object> transientVars, Page odfRootPage, Program program, SubProgram subProgram, Course course) throws Exception
    {
        Page liveOdfRootPage = _resolver.resolveById(odfRootPage.getId());
        
        try
        {
            Page abstractProgramPage = null;
            if (subProgram != null)
            {
                // Page of subprogram
                abstractProgramPage = _odfPageResolver.getSubProgramPage(liveOdfRootPage, subProgram, program);
            }
            else
            {
                abstractProgramPage = _odfPageResolver.getProgramPage(liveOdfRootPage, program);
            }
            
            if (abstractProgramPage != null)
            {
                String abstractProgramPageId = abstractProgramPage.getId();
                getLogger().info("Updating Solr page document with id: {}", abstractProgramPageId);
                
                // Index recursively
                _solrPageIndexer.indexPage(abstractProgramPageId, WebConstants.LIVE_WORKSPACE, true, true);
                
                // No need to index course page(s) as they are already indexed in line above as we index recursively 
//                if (course != null)
//                {
//                    // Retrieve course in the live workspace
//                    Page liveCoursePage = _odfPageResolver.getCoursePage(liveOdfRootPage, course, program);
//                    if (liveCoursePage != null)
//                    {
//                        // Index course page recursively
//                        _solrPageIndexer.indexPage(liveCoursePage.getId(), WebConstants.LIVE_WORKSPACE, true, true, false);
//                    }
//                    else
//                    {
//                        getLogger().warn("Can not find live course page for course '{}' ({}) and parent program '{}' ({})", course.getTitle(), course.getId(), program.getTitle(), program.getId());
//                    }
//                }
            }
        }
        catch (UnknownAmetysObjectException e)
        {
            // Page is not synchronized
        }
    }
}
