/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation.solr;

import java.util.Collection;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Observer;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.web.WebConstants;
import org.ametys.web.indexing.observation.SolrPageContentModifiedObserver;
import org.ametys.web.repository.page.Page;

/**
 * This {@link Observer} observes ODF content modification
 * in order to synchronize the solr index of referencing pages
 */
public class SolrOdfContentModifiedObserver extends SolrPageContentModifiedObserver implements Contextualizable
{
    private OdfPageResolver _odfPageResolver;
    
    private Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfPageResolver = (OdfPageResolver) manager.lookup(OdfPageResolver.ROLE);
    }
    
    @Override
    protected void _updateIndexReferencingPages (String contentId, boolean recursively) throws Exception
    {
        // indicate that we are currently indexing, to allow further optimizations
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(AbstractSolrODFObserver.REQUEST_ATTRIBUTE_INDEXING, true);

        Content content = _resolver.resolveById(contentId);
        
        if (content instanceof ProgramItem)
        {
            Collection<Page> referencingPages = _odfPageResolver.getReferencingPages((ProgramItem) content);
            
            for (Page page : referencingPages)
            {
                // Reindex page in default workspace (its attachments are still indexed, so no need to index them)
                _solrPageIndexer.reindexPage(page.getId(), RepositoryConstants.DEFAULT_WORKSPACE, recursively, false);
                // Reindex page in live workspace if exists (if page was not in live before, its attachments were not either, so index them)
                _solrPageIndexer.reindexPage(page.getId(), WebConstants.LIVE_WORKSPACE, recursively, true);
            }
        }
    }
}
