/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.restrictions;

import java.util.Collections;
import java.util.List;

import org.ametys.odf.program.Program;
import org.ametys.plugins.odfweb.restrictions.rules.OdfAndRestrictionRule;
import org.ametys.plugins.odfweb.restrictions.rules.OdfRestrictionRule;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * A restriction on the ODF programs of a site.
 */
public class OdfProgramRestriction
{
    private final String _id;
    private final I18nizableText _label;
    private final OdfRestrictionRule _rule;
    
    /**
     * Odf restriction constructor
     * @param id Unique id of the restriction
     * @param label The restriction label
     * @param rule The rule of this restriction
     */
    public OdfProgramRestriction(String id, I18nizableText label, OdfRestrictionRule rule)
    {
        _id = id;
        _label = label;
        _rule = rule;
        
        if (_rule == null)
        {
            throw new IllegalArgumentException("At least one non-null rule is required.");
        }
    }
    
    /**
     * Odf restriction constructor
     * @param id Unique id of the restriction
     * @param label The restriction label
     * @param rules The set of rule of this restriction
     */
    public OdfProgramRestriction(String id, I18nizableText label, List<OdfRestrictionRule> rules)
    {
        this(id, label, _extractMainRule(rules));
    }
    
    private static OdfRestrictionRule _extractMainRule(List<OdfRestrictionRule> rules)
    {
        rules.removeAll(Collections.<OdfRestrictionRule>singleton(null));
        
        OdfRestrictionRule rule = null;
        if (rules.size() > 1)
        {
            rule = new OdfAndRestrictionRule(rules);
        }
        else if (!rules.isEmpty())
        {
            rule = rules.iterator().next();
        }
        else
        {
            throw new IllegalArgumentException("At least one non-null rule is required.");
        }
        
        return rule;
    }
    
    /**
     * Id getter
     * @return The id
     */
    public String getId()
    {
        return _id;
    }
    
    /**
     * Label getter
     * @return The label
     */
    public I18nizableText getLabel()
    {
        return _label;
    }
    
    /**
     * Determines if the current restriction contains the program
     * @param program The program to check
     * @return <code>true</code> if the program is part of the restrictions
     */
    public boolean contains(Program program)
    {
        return _rule.contains(program);
    }
    
    /**
     * Translate the restriction into an expression that can be used for repository query.
     * @return The restriction expression
     */
    public Expression getExpression()
    {
        return _rule.getExpression();
    }
    
    /**
     * Indicate if this restriction is related to orgunit
     * @return true if it is the case
     */
    public boolean hasOrgunitRestrictions()
    {
        return _rule.hasOrgunitRestrictions();
    }
}
