/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb;

import java.net.URI;
import java.util.Arrays;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.URIResolver;
import org.ametys.core.util.URIUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.runtime.config.Config;
import org.ametys.web.URIPrefixHandler;
import org.ametys.web.content.GetSiteAction;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.SiteManager;

/**
 * {@link URIResolver} for a ODF Content.
 */
public class OdfURIResolver extends org.ametys.odf.OdfURIResolver implements Contextualizable
{
    private Context _context;

    private SiteManager _siteManager;

    private OdfPageResolver _odfPageResolver;

    private RenderingContextHandler _renderingContextHandler;

    private URIPrefixHandler _prefixHandler;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfPageResolver = (OdfPageResolver) manager.lookup(OdfPageResolver.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
        _prefixHandler = (URIPrefixHandler) manager.lookup(URIPrefixHandler.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public String resolve(String uri, boolean download, boolean absolute, boolean internal)
    {
        Request request = ContextHelper.getRequest(_context);
        String currentSite = (String) request.getAttribute(GetSiteAction.OVERRIDE_SITE_REQUEST_ATTR);
        if (currentSite == null)
        {
            currentSite = (String) request.getAttribute("site");
        }
        if (currentSite == null)
        {
            currentSite = (String) request.getAttribute("siteName");
        }

        // Handle URI like content://UUID or
        // courseContent://UUID;programContent://UUID or
        // courseContent://UUID;courseContent://UUID;programContent://UUID
        int i = uri.indexOf(';');
        String contentId = uri;
        if (i != -1)
        {
            contentId = uri.substring(0, i);
        }

        Content content = _resolver.resolveById(contentId);

        Page odfRootPage = _odfPageResolver.getOdfRootPage(currentSite, content.getLanguage(), ((ProgramItem) content).getCatalog());

        RenderingContext context = _renderingContextHandler.getRenderingContext();

        Page page = null;

        if (content instanceof Program)
        {
            page = _odfPageResolver.getProgramPage((Program) content, odfRootPage.getSiteName());
        }
        else if (content instanceof SubProgram)
        {
            page = _getSubProgramPage((SubProgram) content, odfRootPage, uri.split(";"));
        }
        else if (content instanceof Course)
        {
            page = _getCoursePage((Course) content, odfRootPage, uri.split(";"));
        }
        
        if (page == null)
        {
            // No page found, back to content URI
            String siteName = Config.getInstance().getValue("odf.web.site.name");
            return _getContentURI(content, siteName, context, absolute, internal);
        }

        // We resolved a page either in the current site or in the ODF default site.
        // The site is the page's site.
        String siteName = page.getSiteName();
        String pagePath = page.getSitemapName() + "/" + page.getPathInSitemap() + ".html";

        try
        {
            if (!siteName.equals(currentSite) && context == RenderingContext.FRONT)
            {
                String url = _siteManager.getSite(siteName).getUrl();
                return url + "/" + pagePath;
            }

            if (!(context == RenderingContext.BACK))
            {
                StringBuilder result = new StringBuilder();

                if (internal)
                {
                    result.append("cocoon://").append(siteName);
                }
                else if (absolute)
                {
                    result.append(_prefixHandler.getAbsoluteUriPrefix(siteName));
                }
                else
                {
                    result.append(_prefixHandler.getUriPrefix(siteName));
                }

                result.append("/").append(pagePath);
                
                return URIUtils.encodePath(result.toString());
            }
            else // back
            {
                return "javascript:(function(){parent.Ametys.tool.ToolsManager.openTool('uitool-page', {id:'" + page.getId() + "'});})()";
            }
        }
        catch (Exception e)
        {
            throw new IllegalStateException(e);
        }
    }
    
    private Page _getSubProgramPage(SubProgram subProgram, Page odfRootPage, String[] paths)
    {
        if (odfRootPage == null)
        {
            // There is no ODF root page
            return null;
        }

        if (paths.length > 2)
        {
            // Possible paths are :
            // [subProgramContent://UUID1, programContent://UUID1]
            // [subProgramContent://UUID1, subProgramContent://UUID1]
            // [subprogramContent://UUID, subprogramContent://UUID, subprogramContent://UUID]
            // [subprogramContent://UUID, subprogramContent://UUID, programContent://UUID]

            return _odfPageResolver.getSubProgramPage(odfRootPage, subProgram, Arrays.asList(paths).reversed(), true);
        }
        else
        {
            // [subProgramContent://UUID1]
            return _odfPageResolver.getSubProgramPage(odfRootPage, subProgram, null);
        }
    }

    private Page _getCoursePage(Course course, Page odfRootPage, String[] paths)
    {
        if (odfRootPage == null)
        {
            // There is no ODF root page
            return null;
        }
        
        if (paths.length >= 2)
        {
            // Possible paths are :
            // [courseContent://UUID1, courseContent://UUID1]
            // [courseContent://UUID1, programContent://UUID1]
            // [courseContent://UUID1, subProgramContent://UUID1]
            // [courseContent://UUID1, courseContent://UUID2, ..., courseContent://UUID3, (sub)programContent://UUID]
            // [courseContent://UUID1, courseContent://UUID2, ..., courseContent://UUID3]
            // [courseContent://UUID1, subprogramContent://UUID2, ..., (sub)programContent://UUID3]

            return _odfPageResolver.getCoursePage(odfRootPage, course, Arrays.asList(paths).subList(1, paths.length).reversed(), true);
        }
        else
        {
            // [courseContent://UUID1]
            return _odfPageResolver.getCoursePage(odfRootPage, course, (AbstractProgram) null);
        }
    }

    /**
     * Get the content URI
     * 
     * @param content The content
     * @param siteName The site name
     * @param context The rendering context
     * @param absolute true if the url must be absolute
     * @param internal true to get an internal URI.
     * @return the content URI
     */
    protected String _getContentURI(Content content, String siteName, RenderingContext context, boolean absolute, boolean internal)
    {
        try
        {
            if (context == RenderingContext.FRONT)
            {
                // In FO mode do not give the content URI
                return "";
            }
            else if (!(context == RenderingContext.BACK))
            {
                StringBuilder result = new StringBuilder();

                if (internal)
                {
                    result.append("cocoon://").append(siteName);
                }
                else if (absolute)
                {
                    result.append(_prefixHandler.getAbsoluteUriPrefix());
                }
                else
                {
                    result.append(_prefixHandler.getUriPrefix());
                }
                result.append("/").append("_wrapped-content.html");
                String query = "contentId=" + content.getId() + "&userLocale=" + content.getLanguage() + "&siteName=" + siteName;
                return new URI(null, null, result.toString(), query, null).toASCIIString();
            }
            else
            {
                return "javascript:(function(){parent.Ametys.tool.ToolsManager.openTool('uitool-content', {id:'" + content.getId() + "'});})()";
            }
        }
        catch (Exception e)
        {
            throw new IllegalStateException(e);
        }
    }
}
