/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.cart;

import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.serialization.ZipArchiveSerializer;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;

/**
 * Generate the list of PDF file to include in a zip for downloading cart 
 */
public class CartDownloadGenerator extends ServiceableGenerator
{
    private ODFCartManager _cartManager;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _cartManager = (ODFCartManager) smanager.lookup(ODFCartManager.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String[] itemIds = request.getParameterValues("itemId");
        String siteName = request.getParameter("siteName");
        
        StringBuilder sb = new StringBuilder();
        sb.append("cocoon:/")
            .append(siteName)
            .append("/_content/");
        
        String downloadUriPrefix = sb.toString();
        
        contentHandler.startDocument();
        contentHandler.startPrefixMapping("zip", ZipArchiveSerializer.ZIP_NAMESPACE);
        XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        
        Set<Content> contents = Arrays.stream(itemIds)
            .map(i -> _cartManager.getCartItem(i))
            .filter(Objects::nonNull)
            .map(item -> item.getContent())
            .collect(Collectors.toSet());
        
        for (Content content : contents)
        {
            String contentName = content.getName();
            
            AttributesImpl zipAttrs = new AttributesImpl();
            zipAttrs.addAttribute("", "name", "name", "CDATA", content.getTitle() + ".pdf");
            zipAttrs.addAttribute("", "src", "src", "CDATA", downloadUriPrefix + contentName + ".pdf");
            XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry", zipAttrs);
            XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry");
        }
        
        XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        contentHandler.endPrefixMapping("zip");
        contentHandler.endDocument();
    }
}
