/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation.solr;

import java.util.Map;

import javax.jcr.Repository;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.ObservationConstants;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.site.SiteConfigurationExtensionPoint;

/**
 * {@link Observer} unindexing pages corresponding to {@link Program}s that are not displayed by the site anymore
 * given current ODF restrictions.
 */
public class SolrContentValidatedPart1Observer extends AbstractSolrODFObserver
{
    /** The site configuration extension point. */
    protected SiteConfigurationExtensionPoint _siteConf;
    /** The JCR repository */
    protected Repository _repository;
    /** The ODF page handler */
    protected OdfPageHandler _odfPageHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _siteConf = (SiteConfigurationExtensionPoint) manager.lookup(SiteConfigurationExtensionPoint.ROLE);
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _odfPageHandler = (OdfPageHandler) manager.lookup(OdfPageHandler.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED);
    }
    
    @Override
    public int getPriority()
    {
        // Before live synchronization.
        return MAX_PRIORITY + 500;
    }
    
    @Override
    protected String _workspaceToUse()
    {
        return WebConstants.LIVE_WORKSPACE;
    }
    
    @Override
    protected void _updateIndex(Event event, Map<String, Object> transientVars, Page odfRootPage, Program program, SubProgram subProgram, Course course) throws Exception
    {
        Page liveOdfRootPage = _resolver.resolveById(odfRootPage.getId());
        
        // Test the program in its default version (with the new orgunit reference,
        // as we are before live synchronization) but use the OrgUnit structure
        // from the live workspace (forced above), as we want the structure
        // which is present in Solr indexes.
        
        Request request = ContextHelper.getRequest(_context);
        
        // Retrieve current workspace
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        boolean isValid;
        try
        {
            // is part of restrictions in default workspace ?
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, RepositoryConstants.DEFAULT_WORKSPACE);
            isValid = _odfPageHandler.isValidRestriction(liveOdfRootPage, program);
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
        
        String programId = program.getId();
        if (isValid || !_resolver.hasAmetysObjectForId(programId))
        {
            return;
        }
        
        // Retrieve program in the live workspace
        Program liveProgram = _resolver.resolveById(programId);
        
        Page liveAbstractProgramPage = null;
        if (subProgram != null)
        {
            // Retrieve subprogram in the live workspace
            SubProgram liveSubProgram = _resolver.resolveById(subProgram.getId());
            // Pages of subprogram
            liveAbstractProgramPage = _odfPageResolver.getSubProgramPage(liveOdfRootPage, liveSubProgram, liveProgram);
        }
        else // program is the concerned content by the event
        {
            liveAbstractProgramPage = _odfPageResolver.getProgramPage(liveOdfRootPage, liveProgram);
        }
        
        // If the page exists in the live whereas the program is outside the restrictions, unindex it.
        // (the program won't be displayed in this site anymore after live synchronization).
        if (liveAbstractProgramPage != null)
        {
            String abstractProgramPageId = liveAbstractProgramPage.getId();
            getLogger().info("Removing Solr page document with id: {}", abstractProgramPageId);
            
            // Unindex recursively
            _solrPageIndexer.unindexPage(abstractProgramPageId, WebConstants.LIVE_WORKSPACE, true, true);
        }
    }
}
