/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.restrictions.rules;

import java.util.Collection;
import java.util.Collections;

import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.NotExpression;

/**
 * Not restriction rule. Negate a set of rules.
 */
public class OdfNotRestrictionRule implements OdfRestrictionRule
{
    private final OdfRestrictionRule _rule;
    
    /**
     * Not restriction rule constructor.
     * @param rules The list of rules on which a "not" operator must be applied
     */
    public OdfNotRestrictionRule(Collection<OdfRestrictionRule> rules)
    {
        rules.removeAll(Collections.<OdfRestrictionRule>singleton(null));
        
        if (rules.size() > 1)
        {
            _rule = new OdfAndRestrictionRule(rules);
        }
        else if (!rules.isEmpty())
        {
            _rule = rules.iterator().next();
        }
        else
        {
            throw new IllegalArgumentException("At least one non-null rule is required.");
        }
    }
    
    @Override
    public boolean contains(Program program)
    {
        return !_rule.contains(program);
    }
    
    @Override
    public Expression getExpression()
    {
        Expression expr = _rule.getExpression();
        return expr != null ? new NotExpression(expr) : null;
    }
    
    @Override
    public boolean hasOrgunitRestrictions()
    {
        return _rule.hasOrgunitRestrictions();
    }
}
