/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search.helper;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.odfweb.service.search.criterion.DegreeUniversityCriterionDefinition;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.frontoffice.search.metamodel.SearchServiceCriterionDefinition;
import org.ametys.web.frontoffice.search.metamodel.Searchable;

/**
 * The helper for degree university
 */
public class DegreeUniversityHelper extends AbstractLogEnabled implements Component, Serviceable, PluginAware
{
    /** The avalon role. */
    public static final String ROLE = DegreeUniversityHelper.class.getName();

    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
    
    /** The plugin name */
    protected String _pluginName;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    /**
     * Get all degree contents of type university or not
     * @param isUniversity <code>true</code> to return all degree contents of type university
     * @return the list of degree contents
     */
    public List<Content> getDegrees(boolean isUniversity)
    {
        StringExpression duExpression = new StringExpression(DegreeUniversityCriterionDefinition.ATTRIBUTE_DEGREE_TYPE, Operator.EQ, DegreeUniversityCriterionDefinition.ATTRIBUTE_DEGREE_TYPE_UNIVERSITY_VALUE);
        Expression expr = new AndExpression(
            new ContentTypeExpression(Operator.EQ, OdfReferenceTableHelper.DEGREE),
            isUniversity ? duExpression : new NotExpression(duExpression)
        );
        
        return _resolver.query(QueryHelper.getXPathQuery(null, RepositoryConstants.NAMESPACE_PREFIX + ":content", expr))
            .stream()
            .filter(Content.class::isInstance)
            .map(Content.class::cast)
            .collect(Collectors.toList());
    }
    
    /**
     * Get the degree university criterion definition
     * @param searcheable the searcheable link to the criterion definition
     * @return the criterion definition
     */
    public SearchServiceCriterionDefinition getDegreeUniversityCriterionDefinition(Searchable searcheable)
    {
        ContentType programContentType = _contentTypeExtensionPoint.getExtension(ProgramFactory.PROGRAM_CONTENT_TYPE);
        ElementDefinition degreeDefinition = (ElementDefinition) programContentType.getModelItem(AbstractProgram.DEGREE);
        String name = DegreeUniversityCriterionDefinition.DEGREE_UNIVERSITY_CRITERION_DEFINITION_NAME;
        
        SearchServiceCriterionDefinition criterionDefinition = new DegreeUniversityCriterionDefinition(degreeDefinition, AbstractProgram.DEGREE, programContentType);
        
        criterionDefinition.setName(name);
        criterionDefinition.setPluginName(_pluginName);
        criterionDefinition.setSearchable(searcheable);
        
        return criterionDefinition;
    }
}
