/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.cart;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.cms.repository.Content;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfweb.cart.ODFCartManager.ODFCartItem;
import org.ametys.web.WebHelper;

/**
 * Get the ODF cart items saved in user preferences
 *
 */
public class GetSavedOdfCartItemsAction extends ServiceableAction
{
    private ODFCartManager _cartManager;
    private CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _cartManager = (ODFCartManager) smanager.lookup(ODFCartManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);

        List<Map<String, Object>> items2json = new ArrayList<>();
        
        String siteName = WebHelper.getSiteName(request);
        
        UserIdentity owner = _currentUserProvider.getUser();
        if (owner != null)
        {
            try
            {
                List<ODFCartItem> items = _cartManager.getCartItems(owner, siteName);
                for (ODFCartItem item : items)
                {
                    Map<String, Object> item2json = new HashMap<>();
                    item2json.put("id", item.getId());
                    Content content = item.getContent();
                    item2json.put("contentId", content.getId());
                    item2json.put("title", content.getTitle());
                    
                    Program parentProgram = item.getParentProgram();
                    if (parentProgram != null)
                    {
                        item2json.put("parentProgramId", parentProgram.getId());
                    }
                    items2json.add(item2json);
                }
            }
            catch (UserPreferencesException e)
            {
                getLogger().error("Failed to get cart items for current user " + owner, e);
            }
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("items", items2json);
        result.put("isAnonymous", owner == null);
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }

}
