/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation.solr;

import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.content.indexing.solr.observation.ObserverHelper;
import org.ametys.cms.indexing.IndexingObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfweb.observation.AbstractODFObserver;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.web.indexing.solr.SolrPageIndexer;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Abstract {@link Observer} for synchronizing the Solr indexes.
 */
public abstract class AbstractSolrODFObserver extends AbstractODFObserver implements IndexingObserver
{
    /** Attribute request indicating an ongoing indexation */
    public static final String REQUEST_ATTRIBUTE_INDEXING = "odfweb.solr.indexing";
    
    /** The page indexer. */
    protected SolrPageIndexer _solrPageIndexer;
    /** ODF page resolver */
    protected OdfPageResolver _odfPageResolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _solrPageIndexer = (SolrPageIndexer) manager.lookup(SolrPageIndexer.ROLE);
        _odfPageResolver = (OdfPageResolver) manager.lookup(OdfPageResolver.ROLE);
    }
    
    @Override
    public int getPriority()
    {
        // Will be processed after live synchronization observers
        return MAX_PRIORITY + 3000;
    }
    
    @Override
    protected final void _internalObserve(Event event, Map<String, Object> transientVars, Page odfRootPage, Set<Program> programs, SubProgram subProgram, Course course) throws Exception
    {
        if (ObserverHelper.isNotSuspendedObservationForIndexation())
        {
            // indicate that we are currently indexing, to allow further optimizations
            Request request = ContextHelper.getRequest(_context);
            request.setAttribute(REQUEST_ATTRIBUTE_INDEXING, true);
            
            Sitemap sitemap = odfRootPage.getSitemap();
            String sitemapName = sitemap.getSitemapName();
            
            for (Program program : programs)
            {
                if (StringUtils.equals(sitemapName, program.getLanguage()))
                {
                    _updateIndex(event, transientVars, odfRootPage, program, subProgram, course);
                }
            }
        }
    }
    
    /**
     * Update the index from the observed event.
     * @param event the event.
     * @param transientVars transientVars passed from one Observer to another when processing a single Event.
     * This may allow optimizations between observers.
     * @param odfRootPage the odf root page
     * @param program the target program
     * @param subProgram The subprogram. Can be null
     * @param course the course. Can be null.
     * @throws Exception if an error occurs.
     */
    protected abstract void _updateIndex(Event event, Map<String, Object> transientVars, Page odfRootPage, Program program, SubProgram subProgram, Course course) throws Exception;
    
}
