/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.program.Program;
import org.ametys.odf.tree.OdfClassificationHandler.LevelValue;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.EmptyIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.virtual.VirtualAmetysObjectFactory;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

/**
 * {@link AmetysObjectFactory} handling {@link FirstLevelPage}.
 * This factory is referenced by an ODF root page.
 */
public class FirstLevelPageFactory extends AbstractOdfPageFactory implements VirtualAmetysObjectFactory<Page>
{
    @Override
    public String getScheme()
    {
        return "odfLevel1";
    }
    
    @Override
    public Page getAmetysObjectById(String id) throws AmetysRepositoryException
    {
        // E.g: odfLevel1://XA?rootId=xxx
        String childName = _extractChildName(id);
        Page rootPage = _extractRoot(id);
        
        return _findFirstLevelValueEntry(rootPage, childName)
            .map(entry -> _toFirstLevelPage(rootPage, entry))
            .orElseThrow(() -> 
                new UnknownAmetysObjectException("There's no virtual child page named " + childName + " for parent " + rootPage));
    }
    
    private Page _extractRoot(String id)
    {
        String rootId = StringUtils.substringAfter(id, "?rootId=");
        return _resolver.resolveById(rootId);
    }
    
    private String _extractChildName(String id)
    {
        String rawChildName = StringUtils.substringBetween(id, "://", "?");
        return _odfPageHandler.decodeLevelValue(rawChildName);
    }
    
    private Optional<Entry<String, LevelValue>> _findFirstLevelValueEntry(Page rootPage, String levelCode)
    {
        Map<String, LevelValue> firstLevelValues = _odfPageHandler.getLevel1Values(rootPage);
        return _findFirstLevelValueEntry(firstLevelValues, levelCode);
    }
    
    private Optional<Entry<String, LevelValue>> _findFirstLevelValueEntry(Map<String, LevelValue> firstLevelValues, String levelCode)
    {
        return firstLevelValues.entrySet().stream()
             // entry = (code, title)
            .filter(entry -> entry.getKey().equals(levelCode))
            .findFirst();
    }
    
    private FirstLevelPage _toFirstLevelPage(Page rootPage, Map.Entry<String, LevelValue> firstLevelValueEntry) throws AmetysRepositoryException
    {
        String childName = firstLevelValueEntry.getKey();
        String title = firstLevelValueEntry.getValue().getValue();
        
        VirtualPageConfiguration configuration = getConfiguration();
        
        return new FirstLevelPage(rootPage, configuration, this, childName, title);
    }
    
    private ProgramPage _toProgramPage(Program program, Page root)
    {
        ProgramPageFactory programPageFactory = getProgramPageFactory();
        return new ProgramPage(root, programPageFactory.getConfiguration(), programPageFactory, program, null, null, root);
    }
    
    @Override
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        // E.g: odfLevel1://XA?rootId=xxx
        String childName = _extractChildName(id);
        Page rootPage = _extractRoot(id);
        
        return _hasFirstLevelValue(rootPage, childName);
    }
    
    @Override
    public Page getChild(JCRAmetysObject parent, String childName)
    {
        if (!(parent instanceof Page rootPage))
        {
            throw new IllegalArgumentException("The holder of the ODF virtual pages should be a page.");
        }
        
        String level1Metadata = _odfPageHandler.getLevel1Metadata(rootPage);
        if (StringUtils.isBlank(level1Metadata))
        {
            return _pageCache.getChildProgramPage(rootPage, rootPage, null, null, childName);
        }
        
        Map<String, LevelValue> firstLevelValues = _odfPageHandler.getLevel1Values(rootPage);
        
        // E.g: licence-lmd-XA
        int i = childName.lastIndexOf("-");
        String code = i != -1 ? _odfPageHandler.decodeLevelValue(childName.substring(i + 1)) : _odfPageHandler.decodeLevelValue(childName); // extract first level code
        
        return Optional.of(code)
            .map(firstLevelValues::get)
            .map(LevelValue::getValue)
            .map(title -> new FirstLevelPage(rootPage, getConfiguration(), this, code, title))
            .map(page -> _odfPageHandler.addRedirectIfNeeded(page, childName))
            .orElseThrow(() -> new UnknownAmetysObjectException("There's no virtual child page named " + childName + " for parent " + parent));
    }
    
    @Override
    public AmetysObjectIterable<Page> getChildren(JCRAmetysObject parent)
    {
        if (!(parent instanceof Page rootPage))
        {
            throw new IllegalArgumentException("The holder of the ODF virtual pages should be a page.");
        }
        
        String level1Metadata = _odfPageHandler.getLevel1Metadata(rootPage);
        if (StringUtils.isBlank(level1Metadata))
        {
            List<Page> pages = _pageCache.getPrograms(rootPage, null, null, true)
                    .orElseGet(() -> new EmptyIterable<>())
                    .stream()
                    .map(p -> _toProgramPage(p, rootPage))
                    .collect(Collectors.toList());
            
            return new CollectionIterable<>(pages);
        }
        else
        {
            Map<String, LevelValue> firstLevelValues = _odfPageHandler.getLevel1Values(rootPage);
            Map<String, Map<String, Collection<Program>>> firstLevelCache = _pageCache.getProgramCache(rootPage, true);
            
            List<Page> children = firstLevelCache.keySet().stream()
                    .map(firstLevelCode -> _findFirstLevelValueEntry(firstLevelValues, firstLevelCode).orElse(null))
                    .filter(Objects::nonNull)
                    .map(entry -> _toFirstLevelPage(rootPage, entry))
                    .collect(Collectors.toList());
            
            return new CollectionIterable<>(children);
        }
    }

    @Override
    public boolean hasChild(JCRAmetysObject parent, String childName)
    {
        if (!(parent instanceof Page rootPage))
        {
            throw new IllegalArgumentException("The holder of the ODF virtual pages should be a page.");
        }
        
        String level1Metadata = _odfPageHandler.getLevel1Metadata(rootPage);
        if (StringUtils.isBlank(level1Metadata))
        {
            Optional<Program> program = _pageCache.getProgramFromPageName(rootPage, null, null, childName);
            return program.isPresent();
        }
        else
        {
            return _hasFirstLevelValue(rootPage, childName);
        }
    }
    
    private boolean _hasFirstLevelValue(Page rootPage, String level1Code) throws AmetysRepositoryException
    {
        // FIXME should use page cache, because can return true when page does
        // not exist. Maybe done like this for performance reasons?
        Map<String, LevelValue> firstLevelValues = _odfPageHandler.getLevel1Values(rootPage);
        return firstLevelValues.containsKey(level1Code);
    }
}
