/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.tree.OdfClassificationHandler;
import org.ametys.odf.tree.OdfClassificationHandler.LevelValue;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.odfweb.restrictions.OdfProgramRestriction;
import org.ametys.plugins.odfweb.restrictions.OdfProgramRestrictionManager;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.provider.WorkspaceSelector;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

import com.google.common.collect.ImmutableList;

/**
 * Component providing methods to retrieve ODF virtual pages, such as the ODF root,
 * level 1 and 2 metadata names, and so on.
 */
public class OdfPageHandler extends AbstractLogEnabled implements Component, Initializable, Serviceable
{
    /** The avalon role. */
    public static final String ROLE = OdfPageHandler.class.getName();
    
    /** First level attribute name. */
    public static final String LEVEL1_ATTRIBUTE_NAME = "firstLevel";
    
    /** Second level attribute name. */
    public static final String LEVEL2_ATTRIBUTE_NAME = "secondLevel";
    
    /** Catalog data name. */
    public static final String CATALOG_DATA_NAME = "odf-root-catalog";
    
    /** Content types that are not eligible for first and second level */
    // See ODF-1115 Exclude the mentions enumerator from the list :
    protected static final List<String> NON_ELIGIBLE_CTYPES_FOR_LEVEL = Arrays.asList("org.ametys.plugins.odf.Content.programItem", "odf-enumeration.Mention");
    
    private static final String __ODF_ROOT_PAGES_CACHE = OdfPageHandler.class.getName() + "$odfRootPages";
    private static final String __HAS_ODF_ROOT_CACHE = OdfPageHandler.class.getName() + "$hasOdfRootPage";
    private static final String __PROGRAM_LEVEL_PATH_CACHE = OdfPageHandler.class.getName() + "$programLevelPath";
    private static final String __PROGRAM_RESTRICTION_CACHE = OdfPageHandler.class.getName() + "$programRestriction";
    
    private static final String __ROOT_CACHE_ALL_SITES_KEY = "ALL";
    private static final String __ROOT_CACHE_ALL_SITEMAPS_KEY = "ALL";
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils. */
    protected I18nUtils _i18nUtils;
    
    /** The content type extension point. */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** The ODF Catalog enumeration */
    protected CatalogsManager _catalogsManager;
    
    /** The workspace selector. */
    protected WorkspaceSelector _workspaceSelector;
    
    /** Avalon service manager */
    protected ServiceManager _manager;
    
    /** Restriction manager */
    protected OdfProgramRestrictionManager _odfRestrictionsManager;
    
    /** Content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** Content helper */
    protected ContentHelper _contentHelper;
    
    /** Odf reference table helper */
    protected OdfReferenceTableHelper _odfReferenceTableHelper;
    
    /** Root orgunit provider */
    protected RootOrgUnitProvider _orgUnitProvider;
    
    /** Root orgunit provider */
    protected OdfClassificationHandler _odfClassificationHandler;
    
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _manager = serviceManager;
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _workspaceSelector = (WorkspaceSelector) serviceManager.lookup(WorkspaceSelector.ROLE);
        _catalogsManager = (CatalogsManager) serviceManager.lookup(CatalogsManager.ROLE);
        _odfRestrictionsManager = (OdfProgramRestrictionManager) serviceManager.lookup(OdfProgramRestrictionManager.ROLE);
        _contentTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
        _odfReferenceTableHelper = (OdfReferenceTableHelper) serviceManager.lookup(OdfReferenceTableHelper.ROLE);
        _orgUnitProvider = (RootOrgUnitProvider) serviceManager.lookup(RootOrgUnitProvider.ROLE);
        _odfClassificationHandler = (OdfClassificationHandler) serviceManager.lookup(OdfClassificationHandler.ROLE);
        _cacheManager = (AbstractCacheManager) serviceManager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _cacheManager.createMemoryCache(__ODF_ROOT_PAGES_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ODF_ROOT_PAGES_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_ODF_ROOT_PAGES_DESCRIPTION"),
                true,
                null);
        
        _cacheManager.createMemoryCache(__HAS_ODF_ROOT_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_HAS_ODF_ROOT_PAGE_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_HAS_ODF_ROOT_PAGE_DESCRIPTION"),
                true,
                null);
        
        _cacheManager.createRequestCache(__PROGRAM_LEVEL_PATH_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_PROGRAM_LEVEL_PATH_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_PROGRAM_LEVEL_PATH_DESCRIPTION"),
                false);
        
        _cacheManager.createRequestCache(__PROGRAM_RESTRICTION_CACHE,
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_PROGRAM_RESTRICTION_LABEL"),
                new I18nizableText("plugin.odf-web", "PLUGINS_ODF_WEB_CACHE_PROGRAM_RESTRICTION_DESCRIPTION"),
                false);
    }
    
    /**
     * Get the first ODF root page.
     * @param siteName The site name
     * @param sitemapName The sitemap's name
     * @return a ODF root page or null if not found.
     * @throws AmetysRepositoryException if an error occurs
     */
    public Page getOdfRootPage(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Set<Page> rootPages = getOdfRootPages(siteName, sitemapName);
        return rootPages.isEmpty() ? null : rootPages.iterator().next();
    }
    
    /**
     * Get ODF root page of a specific catalog.
     * @param siteName The site name
     * @param sitemapName The sitemap name
     * @param catalogName The catalog name
     * @return the ODF root page or null if not found.
     * @throws AmetysRepositoryException if an error occurs
     */
    public Page getOdfRootPage(String siteName, String sitemapName, String catalogName) throws AmetysRepositoryException
    {
        String catalogToCompare = catalogName != null ? catalogName : "";
        
        for (Page odfRootPage : getOdfRootPages(siteName, sitemapName))
        {
            if (catalogToCompare.equals(getCatalog(odfRootPage)))
            {
                return odfRootPage;
            }
        }
        
        return null;
    }
    
    /**
     * Get the id of ODF root pages
     * @param siteName The site name
     * @param sitemapName The sitemap name
     * @return The ids of ODF root pages
     * @throws AmetysRepositoryException if an error occurs.
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED) // only retrieve page ids
    public List<String> getOdfRootPageIds(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Set<Page> pages = getOdfRootPages(siteName, sitemapName);
        return pages.stream().map(p -> p.getId()).collect(Collectors.toList());
    }
    
    /**
     * Get the ODF root pages.
     * @param siteName the current site.
     * @param sitemapName the current sitemap/language.
     * @return the ODF root pages
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Set<Page> getOdfRootPages(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Cache<OdfRootPageCacheKey, Set<String>> cache = _getOdfRootPagesCache();
        
        String workspaceName = _workspaceSelector.getWorkspace();
        
        Set<String> rootPageIds = cache.get(OdfRootPageCacheKey.of(workspaceName, Objects.toString(siteName, __ROOT_CACHE_ALL_SITES_KEY), Objects.toString(sitemapName, __ROOT_CACHE_ALL_SITEMAPS_KEY)), item -> {
            return _getOdfRootPages(siteName, sitemapName)
                    .stream()
                    .map(Page::getId)
                    .collect(Collectors.toSet());
        });
        
        return rootPageIds.stream()
                   .map(id -> (Page) _resolver.resolveById(id))
                   .collect(Collectors.toSet());
    }
    
    /**
     * Test if the given site has at least one sitemap with an odf root page.
     * @param site the site to test.
     * @return true if the site has at least one sitemap with an odf root page, false otherwise.
     */
    public boolean hasOdfRootPage(Site site)
    {
        Cache<HasOdfRootPageCacheKey, Boolean> cache = _getHasOdfRootPageCache();
        
        String workspace = _workspaceSelector.getWorkspace();
        
        return cache.get(HasOdfRootPageCacheKey.of(workspace, site.getName()), item -> {
            
            Iterator<Sitemap> sitemaps = site.getSitemaps().iterator();
            while (sitemaps.hasNext())
            {
                String sitemapName = sitemaps.next().getName();
                
                if (!getOdfRootPages(site.getName(), sitemapName).isEmpty())
                {
                    return true;
                }
            }
            
            return false;
        });
    }
    
    /**
     * Determines if the program is part of the site restrictions
     * @param rootPage The ODF root page
     * @param program The program
     * @return <code>true</code> the program is part of the site restrictions
     */
    public boolean isValidRestriction(Page rootPage, Program program)
    {
        Cache<ProgramInRootCacheKey, Boolean> cache = _cacheManager.get(__PROGRAM_RESTRICTION_CACHE);
        
        return cache.get(ProgramInRootCacheKey.of(rootPage.getId(), program.getId()), isValid -> {
            // Check catalog
            if (!program.getCatalog().equals(getCatalog(rootPage)))
            {
                return false;
            }
            
            // Check language
            if (!program.getLanguage().equals(rootPage.getSitemapName()))
            {
                return false;
            }
            
            // Check site restrictions
            OdfProgramRestriction restriction = _odfRestrictionsManager.getRestriction(rootPage);
            if (restriction != null)
            {
                return restriction.contains(program);
            }
            
            return true;
        });
    }
    
    /**
     * Clear the ODF root page cache.
     */
    public void clearRootCache()
    {
        _getOdfRootPagesCache().invalidateAll();
        _getHasOdfRootPageCache().invalidateAll();
    }
    
    /**
     * Clear the ODF root page cache for a given site and language.
     * @param siteName the current site.
     * @param sitemapName the current sitemap/language.
     */
    public void clearRootCache(String siteName, String sitemapName)
    {
        _getOdfRootPagesCache().invalidate(OdfRootPageCacheKey.of(null, siteName, sitemapName));
        _getHasOdfRootPageCache().invalidate(HasOdfRootPageCacheKey.of(null, siteName));
    }
    
    /**
     * Determines if the page is a ODF root page
     * @param page The page to test
     * @return true if the page is a ODF root page
     */
    public boolean isODFRootPage (Page page)
    {
        if (page instanceof JCRAmetysObject)
        {
            try
            {
                JCRAmetysObject jcrPage = (JCRAmetysObject) page;
                Node node = jcrPage.getNode();
                
                if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
                {
                    Value[] values = node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues();
                    
                    boolean hasValue = false;
                    for (int i = 0; i < values.length && !hasValue; i++)
                    {
                        hasValue = FirstLevelPageFactory.class.getName().equals(values[i].getString());
                    }
                    
                    return hasValue;
                }
                else
                {
                    return false;
                }
            }
            catch (RepositoryException e)
            {
                return false;
            }
        }
        
        return false;
        
    }
    
    /**
     * Get the ODF root page.
     * @param siteName the current site.
     * @param sitemapName the current sitemap/language.
     * @return the ODF root page or null if not found.
     * @throws AmetysRepositoryException if an error occurs.
     */
    protected Page _getOdfRootPage(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Expression expression = new VirtualFactoryExpression(FirstLevelPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(siteName, sitemapName, null, expression, null);
        
        AmetysObjectIterable<Page> pages = _resolver.query(query);
        Page page = pages.stream().findFirst().orElse(null);
        
        return page;
    }
    
    /**
     * Get the ODF root page.
     * @param siteName the current site.
     * @param sitemapName the current sitemap/language.
     * @return the ODF root page or null if not found.
     * @throws AmetysRepositoryException if an error occurs.
     */
    protected Set<Page> _getOdfRootPages(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Expression expression = new VirtualFactoryExpression(FirstLevelPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(siteName, sitemapName, null, expression, null);
        
        return _resolver.<Page>query(query).stream().collect(Collectors.toSet());
    }
    
    /**
     * Get the catalog value of the ODF root page
     * @param rootPage The ODF root page
     * @return the catalog value
     */
    public String getCatalog (Page rootPage)
    {
        return rootPage.getValue(CATALOG_DATA_NAME, StringUtils.EMPTY);
    }
    
    /**
     * Get the first level metadata name.
     * @param siteName the site name.
     * @param sitemapName the sitemap name.
     * @param catalog the current selected catalog.
     * @return the first level metadata name.
     */
    public String getLevel1Metadata(String siteName, String sitemapName, String catalog)
    {
        Page rootPage = getOdfRootPage(siteName, sitemapName, catalog);
        
        return getLevel1Metadata(rootPage);
    }
    
    /**
     * Get the first level metadata name.
     * @param rootPage the ODF root page.
     * @return the first level metadata name.
     */
    public String getLevel1Metadata(Page rootPage)
    {
        return rootPage.getValue(LEVEL1_ATTRIBUTE_NAME);
    }
    
    /**
     * Get the second level metadata name.
     * @param siteName the site name.
     * @param sitemapName the sitemap name.
     * @param catalog the current selected catalog.
     * @return the second level metadata name.
     */
    public String getLevel2Metadata(String siteName, String sitemapName, String catalog)
    {
        Page rootPage = getOdfRootPage(siteName, sitemapName, catalog);
        
        return getLevel2Metadata(rootPage);
    }
    
    /**
     * Get the second level metadata name.
     * @param rootPage the ODF root page.
     * @return the second level metadata name.
     */
    public String getLevel2Metadata(Page rootPage)
    {
        return rootPage.getValue(LEVEL2_ATTRIBUTE_NAME);
    }
    
    /**
     * Get the first level metadata values (with translated label).
     * @param siteName the site name.
     * @param sitemapName the sitemap name.
     * @param catalog the current selected catalog.
     * @return the first level metadata values.
     */
    public Map<String, LevelValue> getLevel1Values(String siteName, String sitemapName, String catalog)
    {
        Page rootPage = getOdfRootPage(siteName, sitemapName, catalog);
        
        return getLevel1Values(rootPage);
    }
    
    /**
     * Get the level value of a program by extracting and transforming the raw program value at the desired metadata path
     * @param program The program
     * @param levelMetaPath The desired metadata path that represent a level
     * @return The final level value
     */
    public String getProgramLevelValue(Program program, String levelMetaPath)
    {
        List<String> programLevelValue = _odfClassificationHandler.getProgramLevelValues(program, levelMetaPath);
        return programLevelValue.isEmpty() ? null : programLevelValue.get(0);
    }
    
    /**
     * Get the first level value of a program by extracting and transforming the raw program value
     * @param rootPage The root page
     * @param program The program
     * @return The final level value or <code>null</code> if not found
     */
    public String getProgramLevel1Value(Page rootPage, Program program)
    {
        String level1Metadata = getLevel1Metadata(rootPage);
        if (StringUtils.isNotBlank(level1Metadata))
        {
            List<String> programLevelValue = _odfClassificationHandler.getProgramLevelValues(program, level1Metadata);
            return programLevelValue.isEmpty() ? null : programLevelValue.get(0);
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Get the second level value of a program by extracting and transforming the raw program value
     * @param rootPage The root page
     * @param program The program
     * @return The final level value or <code>null</code> if not found
     */
    public String getProgramLevel2Value(Page rootPage, Program program)
    {
        String level2Metadata = getLevel2Metadata(rootPage);
        if (StringUtils.isNotBlank(level2Metadata))
        {
            List<String> programLevelValue = _odfClassificationHandler.getProgramLevelValues(program, level2Metadata);
            return programLevelValue.isEmpty() ? null : programLevelValue.get(0);
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Get the orgunit identifier given an uai code
     * @param rootPage Odf root page
     * @param uaiCode The uai code
     * @return The orgunit id or null if not found
     */
    public String getOrgunitIdFromUaiCode(Page rootPage, String uaiCode)
    {
        return _odfClassificationHandler.getOrgunitIdFromUaiCode(rootPage.getSitemapName(), uaiCode);
    }
    
    /**
     * Get the programs available for a ODF root page, taking account the site's restrictions
     * @param rootPage The ODF root page
     * @param level1 filters results with a level1 value. Can be null.
     * @param level2 filters results with a level2 value. Can be null.
     * @param programCode expected program code. Can be null.
     * @param programName expected program name. Can be null.
     * @return an iterator over resulting programs
     */
    public AmetysObjectIterable<Program> getProgramsWithRestrictions(Page rootPage, String level1, String level2, String programCode, String programName)
    {
        return getProgramsWithRestrictions(rootPage, getLevel1Metadata(rootPage), level1, getLevel2Metadata(rootPage), level2, programCode, programName);
    }

    /**
     * Get the programs available for a ODF root page, taking account the site's restrictions
     * @param rootPage The ODF root page
     * @param level1Metadata metadata name for first level
     * @param level1 filters results with a level1 value. Can be null.
     * @param level2Metadata metadata name for second level
     * @param level2 filters results with a level2 value. Can be null.
     * @param programCode expected program code. Can be null.
     * @param programName expected program name. Can be null.
     * @return an iterator over resulting programs
     */
    public AmetysObjectIterable<Program> getProgramsWithRestrictions(Page rootPage, String level1Metadata, String level1, String level2Metadata, String level2, String programCode, String programName)
    {
        OdfProgramRestriction restriction = _odfRestrictionsManager.getRestriction(rootPage);
        return _odfClassificationHandler.getPrograms(getCatalog(rootPage), rootPage.getSitemapName(), level1Metadata, level1, level2Metadata, level2, programCode, programName, restriction == null ? null : ImmutableList.of(restriction.getExpression()));
    }
    
    /**
     * Get the first level metadata values (with translated label)
     * @param rootPage the ODF root page.
     * @return the first level metadata values. Can be empty if there is no level1 attribute on root page.
     */
    public Map<String, LevelValue> getLevel1Values(Page rootPage)
    {
        String level1Value = getLevel1Metadata(rootPage);
        if (StringUtils.isNotBlank(level1Value))
        {
            return _odfClassificationHandler.getLevelValues(level1Value, rootPage.getSitemapName());
        }
        else
        {
            return Collections.EMPTY_MAP;
        }
    }
    
    /**
     * Get the second level metadata values (with translated label).
     * @param siteName the site name.
     * @param sitemapName the sitemap name.
     * @param catalog the current selected catalog.
     * @return the second level metadata values.
     */
    public Map<String, LevelValue> getLevel2Values(String siteName, String sitemapName, String catalog)
    {
        Page rootPage = getOdfRootPage(siteName, sitemapName, catalog);
        
        return getLevel2Values(rootPage);
    }
    
    /**
     * Get the second level metadata values (with translated label).
     * @param rootPage the ODF root page.
     * @return the second level metadata values. Can be empty if there is no level2 attribute on root page.
     */
    public Map<String, LevelValue> getLevel2Values(Page rootPage)
    {
        String level2Value = getLevel2Metadata(rootPage);
        if (StringUtils.isNotBlank(level2Value))
        {
            return _odfClassificationHandler.getLevelValues(level2Value, rootPage.getSitemapName());
        }
        else
        {
            return Collections.EMPTY_MAP;
        }
    }

    /**
     * Encode level value to be use into a URI.
     * @param value The raw value
     * @return the encoded value
     */
    public String encodeLevelValue(String value)
    {
        String encodedValue = StringUtils.replace(value, "-", "@2D");
        encodedValue = StringUtils.replace(encodedValue, "/", "@2F");
        encodedValue = StringUtils.replace(encodedValue, ":", "@3A");
        encodedValue = StringUtils.replace(encodedValue, "?", "@3F");
        return URIUtils.encodePathSegment(encodedValue);
    }
    
    /**
     * Decode level value used in a URI
     * @param value The encoded value
     * @return the decoded value
     */
    public String decodeLevelValue(String value)
    {
        String decodedValue =  URIUtils.decode(value);
        decodedValue = StringUtils.replace(decodedValue, "@3F", "?");
        decodedValue = StringUtils.replace(decodedValue, "@3A", ":");
        decodedValue = StringUtils.replace(decodedValue, "@2F", "/");
        return StringUtils.replace(decodedValue, "@2D", "-");
    }
    
    /**
     * Returns the page's name of a {@link ProgramItem}.
     * Only {@link AbstractProgram} and {@link Course} can have a page.
     * @param item The program item
     * @return The page's name
     * @throws IllegalArgumentException if the program item is not a {@link AbstractProgram} nor a {@link Course}.
     */
    public String getPageName (ProgramItem item)
    {
        if (item instanceof AbstractProgram || item instanceof Course)
        {
            String filteredTitle = "";
            try
            {
                filteredTitle = NameHelper.filterName(((Content) item).getTitle());
            }
            catch (IllegalArgumentException e)
            {
                // title does not match the expected regular expression : ^([0-9-_]*)[a-z].*$, use default title
                if (item instanceof Program)
                {
                    filteredTitle = "program";
                }
                else if (item instanceof SubProgram)
                {
                    filteredTitle = "subprogram";
                }
                else if (item instanceof Course)
                {
                    filteredTitle = "course";
                }
            }
            
            return filteredTitle + "-" + item.getCode();
        }
        else
        {
            throw new IllegalArgumentException("Illegal program item : no page can be associated for a program item of type " + item.getClass().getName());
        }
    }

    /**
     * Get the eligible enumerated attribute definitions for ODF page level
     * @return the eligible attribute definitions
     */
    public Map<String, ModelItem> getEligibleAttributesForLevel()
    {
        return _odfClassificationHandler.getEligibleAttributesForLevel();
    }

    /**
     * Get the ODF catalogs
     * @return the ODF catalogs
     */
    public Map<String, I18nizableText> getCatalogs()
    {
        return _odfClassificationHandler.getCatalogs();
    }

    /**
     * Get the enumerated attribute definitions for the given content type.
     * Attribute with enumerator or content attribute are considered as enumerated
     * @param programContentTypeId The content type's id
     * @param allowMultiple <code>true</code> true to allow multiple attributes
     * @return The definitions of enumerated attributes
     */
    public Map<String, ModelItem> getEnumeratedAttributes(String programContentTypeId, boolean allowMultiple)
    {
        return _odfClassificationHandler.getEnumeratedAttributes(programContentTypeId, allowMultiple);
    }
    
    /**
     * Compute the path from the root odf page, representing the first and second level pages.
     * @param rootPage The odf root page
     * @param parentProgram The program to compute
     * @return the path, can be empty if no levels defined, and null if the parent program does not have values for levels attributes
     */
    public String computeLevelsPath(Page rootPage, Program parentProgram)
    {
        Cache<ProgramInRootCacheKey, String> levelCache = _cacheManager.get(__PROGRAM_LEVEL_PATH_CACHE);
        
        return levelCache.get(ProgramInRootCacheKey.of(rootPage.getId(), parentProgram.getId()), item -> {
            // Level 1 is defined => Check the value
            if (getLevel1Metadata(rootPage) != null)
            {
                String level1 = getProgramLevel1Value(rootPage, parentProgram);
                
                // Value for level 1 is defined => Check for the second level
                if (StringUtils.isNotBlank(level1))
                {
                    // Level 2 is defined => Check the value
                    if (getLevel2Metadata(rootPage) != null)
                    {
                        String level2 = getProgramLevel2Value(rootPage, parentProgram);
                        
                        // Value for level 2 is defined => Return the level 2 page
                        if (StringUtils.isNotBlank(level2))
                        {
                            Page secondLevelPage = findSecondLevelPage(rootPage, level1, level2);
                            return secondLevelPage.getParent().getName() + "/" + secondLevelPage.getName();
                        }
                        
                        // Value for level 2 is not defined => Return null
                        return null;
                    }
                    
                    // Level 2 is not defined => Return the level 1 page
                    return findFirstLevelPage(rootPage, level1).getName();
                }

                // Value for level 1 is not defined => Return null
                return null;
            }
            
            // Level 1 is not defined => Return an empty path, all pages are on the root page
            return StringUtils.EMPTY;
        });
    }
    
    /**
     * Build the level 1 identifier
     * @param rootPage The ODF root page
     * @param level1Value The level1 name
     * @return the identifier beginning by odfLevel1://...
     */
    public String buildLevel1Id(Page rootPage, String level1Value)
    {
        // E.g: odfLevel1://XA?rootId=...
        return "odfLevel1://" + encodeLevelValue(level1Value) + "?rootId=" + rootPage.getId();
    }
    
    /**
     * Build the level 2 identifier
     * @param rootPage The ODF root page
     * @param level1Value The level1 name
     * @param level2Value The level2 name
     * @return the identifier beginning by odfLevel2://...
     */
    public String buildLevel2Id(Page rootPage, String level1Value, String level2Value)
    {
        // E.g: odfLevel2://XA/ALL?rootId=...
        return "odfLevel2://" + encodeLevelValue(level1Value) + "/" + encodeLevelValue(level2Value) + "?rootId=" + rootPage.getId();
    }
    
    /**
     * Get the first level page from the given root page with the level 1 value.
     * @param rootPage The odf root page
     * @param level1Value The first level value
     * @return a first level page
     */
    public FirstLevelPage findFirstLevelPage(Page rootPage, String level1Value)
    {
        // Calculate the real path
        return _resolver.resolveById(buildLevel1Id(rootPage, level1Value));
    }
    
    /**
     * Get the second level page from the given root page with the level 1 and level 2 values.
     * @param rootPage The odf root page
     * @param level1Value The first level value
     * @param level2Value The second level value
     * @return a second level page
     */
    public SecondLevelPage findSecondLevelPage(Page rootPage, String level1Value, String level2Value)
    {
        // Calculate the real path
        return _resolver.resolveById(buildLevel2Id(rootPage, level1Value, level2Value));
    }
    
    /**
     * Add an intermediate redirect page if the called page name doesn't match the real page name.
     * @param page The page
     * @param name The called page name
     * @return The page maybe included in a {@link RedirectPage}
     */
    public Page addRedirectIfNeeded(Page page, String name)
    {
        if (!name.equals(URIUtils.decode(page.getName())))
        {
            getLogger().warn("Redirect path '{}' to '{}' page", name, page.getName());
            return new RedirectPage(page instanceof RedirectPage redirectPage ? redirectPage.getRedirectPage() : page);
        }
        return page;
    }
    
    /**
     * Explore the queue path if it is not empty
     * @param page The root page to explore
     * @param queuePath The path
     * @return The child page, or given page if queue path is empty
     */
    public Page exploreQueuePath(Page page, String queuePath)
    {
        if (StringUtils.isNotEmpty(queuePath))
        {
            return page.getChild(queuePath);
        }
        return page;
    }
    
    private Cache<OdfRootPageCacheKey, Set<String>> _getOdfRootPagesCache()
    {
        return _cacheManager.get(__ODF_ROOT_PAGES_CACHE);
    }
    
    private Cache<HasOdfRootPageCacheKey, Boolean> _getHasOdfRootPageCache()
    {
        return _cacheManager.get(__HAS_ODF_ROOT_CACHE);
    }
    
    static class OdfRootPageCacheKey extends AbstractCacheKey
    {
        public OdfRootPageCacheKey(String workspaceName, String siteName, String sitemapName)
        {
            super(workspaceName, siteName, sitemapName);
        }
        
        public static OdfRootPageCacheKey of(String workspaceName, String siteName, String sitemapName)
        {
            return new OdfRootPageCacheKey(workspaceName, siteName, sitemapName);
        }
    }
    
    static class HasOdfRootPageCacheKey extends AbstractCacheKey
    {
        public HasOdfRootPageCacheKey(String workspaceName, String siteName)
        {
            super(workspaceName, siteName);
        }
        
        public static HasOdfRootPageCacheKey of(String workspaceName, String siteName)
        {
            return new HasOdfRootPageCacheKey(workspaceName, siteName);
        }
    }
    
    static class ProgramInRootCacheKey extends AbstractCacheKey
    {
        public ProgramInRootCacheKey(String rootPageId, String programId)
        {
            super(rootPageId, programId);
        }
        
        public static ProgramInRootCacheKey of(String rootPageId, String programId)
        {
            return new ProgramInRootCacheKey(rootPageId, programId);
        }
    }
}
