/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.restrictions.rules;

import java.util.Arrays;
import java.util.Collections;
import java.util.Optional;

import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.type.ElementType;

/**
 * Attribute restriction rule
 */
public class OdfAttributeRestrictionRule implements OdfRestrictionRule
{
    private final String _attributePath;
    private final String _untypedValue;
    
    /**
     * Constructor for rule relating to the value of an attribute
     * @param attributePath The path of the attribute
     * @param untypedValue The attribute untyped value
     */
    public OdfAttributeRestrictionRule(String attributePath, String untypedValue)
    {
        _attributePath = attributePath;
        _untypedValue = untypedValue;
    }
    
    @Override
    public Expression getExpression()
    {
        return new StringExpression(_attributePath, Operator.EQ, _untypedValue);
    }
    
    @Override
    public boolean contains(Program program)
    {
        ElementType type = (ElementType) program.getDefinition(_attributePath).getType();
        Object typedValue = type.castValue(_untypedValue);
        
        return Optional.ofNullable(program.getValue(_attributePath))
                       .map(value -> Arrays.asList(value))
                       .orElse(Collections.EMPTY_LIST)
                       .stream()
                       .anyMatch(value -> value.equals(typedValue));
    }
    
    @Override
    public boolean hasOrgunitRestrictions()
    {
        return false;
    }
}
