/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.search.advanced.AbstractTreeNode;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.ConstantNilScoreQuery;
import org.ametys.cms.search.query.ContentTypeQuery;
import org.ametys.cms.search.query.IsolateQuery;
import org.ametys.cms.search.query.JoinQuery;
import org.ametys.cms.search.query.MaxScoreOrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.odfweb.service.search.helper.DegreeUniversityHelper;
import org.ametys.web.frontoffice.search.instance.model.SearchServiceCriterion;
import org.ametys.web.frontoffice.search.metamodel.AdditionalParameterValueMap;
import org.ametys.web.frontoffice.search.metamodel.Returnable;
import org.ametys.web.frontoffice.search.metamodel.SearchServiceCriterionDefinition;
import org.ametys.web.frontoffice.search.metamodel.Searchable;
import org.ametys.web.frontoffice.search.metamodel.impl.AbstractContentBasedSearchable;

/**
 * {@link Searchable} for {@link Program}s
 */
public class ProgramSearchable extends AbstractContentBasedSearchable
{
    /** Avalon Role */
    public static final String ROLE = ProgramSearchable.class.getName();
    
    /** The additional parameter for indicating if search has to be made also on subprograms */
    public static final String PARAMETER_SEARCH_ON_SUBPROGRAMS = "searchSubprogram";
    
    /** The prefix for program searchable */
    public static final String CRITERION_DEFINITIONS_PREFIX_ID = "ProgramSearchable$";

    /** The prefix for criterion definition in program searcheable */
    public static final String PROGRAM_SEARCHEABLE_CRITERION_DEFINITION_PREFIX = CRITERION_DEFINITIONS_PREFIX_ID + ProgramFactory.PROGRAM_CONTENT_TYPE + "$";
    
    /** The degree university helper */
    protected DegreeUniversityHelper _degreeUniversityHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _degreeUniversityHelper = (DegreeUniversityHelper) manager.lookup(DegreeUniversityHelper.ROLE);
    }
    
    @Override
    protected String associatedContentReturnableRole()
    {
        return ProgramReturnable.ROLE;
    }
    
    @Override
    protected String getCriterionDefinitionPrefix()
    {
        return CRITERION_DEFINITIONS_PREFIX_ID;
    }
    
    @Override
    public Collection<Returnable> relationsWith()
    {
        return Arrays.asList(_associatedContentReturnable);
    }
    
    @Override
    protected Set<String> getContentTypeIds(AdditionalParameterValueMap additionalParameterValues)
    {
        return Set.of(ProgramFactory.PROGRAM_CONTENT_TYPE);
    }
    
    @Override
    public Optional<Query> joinQuery(Query queryOnCriterion, SearchServiceCriterionDefinition criterion, Collection<Returnable> returnables, AdditionalParameterValueMap additionalParameters)
    {
        if (returnables.contains(_associatedContentReturnable))
        {
            return Optional.of(queryOnCriterion);
        }
        return Optional.empty();
    }
    
    @Override
    public Query buildQuery(
            AbstractTreeNode<SearchServiceCriterion<?>> criterionTree, 
            Map<String, Object> userCriteria, 
            Collection<Returnable> returnables,
            Collection<Searchable> searchables, 
            AdditionalParameterValueMap additionalParameters, 
            String currentLang, 
            Map<String, Object> contextualParameters)
    {
        Query query = super.buildQuery(criterionTree, userCriteria, returnables, searchables, additionalParameters, currentLang, contextualParameters);
        if (_searchOnSubprograms(additionalParameters))
        {
            // This query search on
            //   - programs matching user criteria
            //   OR
            //   - subprograms matching user criteria
            // This is the same query because it is assumed that programs and subprograms have the same attributes (properties are added for subprogram to match with missing program attributes)
            return new MaxScoreOrQuery(
                        new IsolateQuery(query), 
                        getProgramThroughSubprogramsQuery(query)
                    );
        }
        else
        {
            return query;
        }
    }
    
    private boolean _searchOnSubprograms(AdditionalParameterValueMap additionalParameters)
    {
        return additionalParameters.getValue(PARAMETER_SEARCH_ON_SUBPROGRAMS);
    }
    
    /**
     * Gets the {@link Query} that returns Programs by querying their SubPrograms
     * @param queryOnCriterion The query on SubPrograms
     * @return The {@link Query} that returns Programs
     */
    protected Query getProgramThroughSubprogramsQuery(Query queryOnCriterion)
    {
        Query queryOnSubprograms = new AndQuery(
            new ConstantNilScoreQuery(new ContentTypeQuery(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE)),
            queryOnCriterion
        );
        return new ProgramThroughProgramPartsQuery(queryOnSubprograms);
    }
    
    @Override
    public Collection<SearchServiceCriterionDefinition> getCriteria(AdditionalParameterValueMap additionalParameterValues)
    {
        Collection<SearchServiceCriterionDefinition> criteria = super.getCriteria(additionalParameterValues);
        
        SearchServiceCriterionDefinition degreeUniversityCriterionDefinition = _degreeUniversityHelper.getDegreeUniversityCriterionDefinition(this);
        if (degreeUniversityCriterionDefinition != null)
        {
            criteria.add(degreeUniversityCriterionDefinition);
        }
        
        return criteria;
    }

    /**
     * A {@link Query} that returns Programs, by querying their child ProgramParts
     */
    protected static class ProgramThroughProgramPartsQuery extends JoinQuery
    {
        /**
         * Builds a ProgramThroughProgramPartsQuery
         * @param subQuery The query on ProgramParts
         */
        protected ProgramThroughProgramPartsQuery(Query subQuery)
        {
            super(subQuery, TraversableProgramPart.CHILD_PROGRAM_PARTS);
        }
    }
}
