/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.actions;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.runtime.model.ModelItem;

/**
 * Action providing the list of eligible enumerated metadata for ODF page level
 */
public class GetEligibleMetadataList extends ServiceableAction
{
    /** The ODF root page handler. */
    protected OdfPageHandler _odfRootHandler;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _odfRootHandler = (OdfPageHandler) serviceManager.lookup(OdfPageHandler.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        // Retrieve the list of eligible metadata
        List<Map<String, Object>> attributesInfo = _odfRootHandler.getEligibleAttributesForLevel().entrySet().stream()
            .map(this::getAttributeInfo)
            .collect(Collectors.toList());
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, attributesInfo);
        
        return EMPTY_MAP;
    }
    
    /**
     * Generates a {@link Map} with information about the given attribute
     * The returned information are the attribute's name and label
     * @param attributeEntry The attribute's entry with its path as a key and its definition as a value
     * @return a {@link Map} with the attribute's information
     */
    protected Map<String, Object> getAttributeInfo(Entry<String, ModelItem> attributeEntry)
    {
        Map<String, Object> data = new HashMap<>();
        data.put("name", attributeEntry.getKey());
        data.put("label", attributeEntry.getValue().getLabel());
        return data;
    }
}
